<?php
// users.php - User Management
session_start();
require_once 'includes/config.php';
require_once 'includes/functions.php';

checkAdminLogin();

// Pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$limit = 50;
$search = isset($_GET['search']) ? sanitize_input($_GET['search']) : '';

// Get users
$users = getUsers($page, $limit, $search);
$totalUsers = getUsersCount($search);
$totalPages = ceil($totalUsers / $limit);

// Handle actions
$message = '';
$error = '';

if (isset($_GET['action'])) {
    $userId = $_GET['id'] ?? '';
    $action = $_GET['action'];
    
    $db = getDB();
    
    switch ($action) {
        case 'ban':
            $stmt = $db->prepare("UPDATE app_users SET status = 'banned' WHERE user_id = ?");
            if ($stmt->execute([$userId])) {
                $message = 'User banned successfully';
            } else {
                $error = 'Failed to ban user';
            }
            break;
            
        case 'unban':
            $stmt = $db->prepare("UPDATE app_users SET status = 'active' WHERE user_id = ?");
            if ($stmt->execute([$userId])) {
                $message = 'User unbanned successfully';
            } else {
                $error = 'Failed to unban user';
            }
            break;
            
        case 'delete':
            $stmt = $db->prepare("DELETE FROM app_users WHERE user_id = ?");
            if ($stmt->execute([$userId])) {
                $message = 'User deleted successfully';
            } else {
                $error = 'Failed to delete user';
            }
            break;
            
        case 'make_vip':
            $expiry = date('Y-m-d', strtotime('+30 days'));
            $stmt = $db->prepare("UPDATE app_users SET is_vip = 1, vip_expiry = ? WHERE user_id = ?");
            if ($stmt->execute([$expiry, $userId])) {
                $message = 'User upgraded to VIP successfully';
            } else {
                $error = 'Failed to upgrade user';
            }
            break;
            
        case 'remove_vip':
            $stmt = $db->prepare("UPDATE app_users SET is_vip = 0, vip_expiry = NULL WHERE user_id = ?");
            if ($stmt->execute([$userId])) {
                $message = 'VIP status removed successfully';
            } else {
                $error = 'Failed to remove VIP status';
            }
            break;
    }
}

// Get user statistics
$stats = [
    'total' => getTotalUsers(),
    'active' => getActiveUsers(),
    'vip' => getVipUsers(),
    'banned' => getBannedUsersCount(),
    'today' => getTodayRegisteredUsers()
];

function getBannedUsersCount() {
    $db = getDB();
    $stmt = $db->query("SELECT COUNT(*) as count FROM app_users WHERE status = 'banned'");
    $result = $stmt->fetch();
    return $result['count'] ?? 0;
}

function getTodayRegisteredUsers() {
    $db = getDB();
    $stmt = $db->query("SELECT COUNT(*) as count FROM app_users WHERE DATE(created_at) = CURDATE()");
    $result = $stmt->fetch();
    return $result['count'] ?? 0;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Management - Pink Video Chat Admin</title>
    <?php include 'includes/head.php'; ?>
    <style>
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
        }
        .country-flag {
            width: 20px;
            height: 15px;
            margin-right: 5px;
        }
        .status-badge {
            font-size: 0.75rem;
        }
        .action-dropdown {
            min-width: 150px;
        }
        .filter-box {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container mt-4">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="col-md-9">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h3><i class="fas fa-users me-2"></i> User Management</h3>
                    <div>
                        <button class="btn btn-pink" onclick="exportUsers()">
                            <i class="fas fa-download me-1"></i> Export Users
                        </button>
                    </div>
                </div>
                
                <?php if ($message): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <?php echo $message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <?php echo $error; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <!-- User Statistics -->
                <div class="row mb-4">
                    <div class="col-md-2 col-6 mb-3">
                        <div class="card stat-card bg-primary text-white">
                            <div class="card-body text-center py-3">
                                <i class="fas fa-users fa-lg mb-2"></i>
                                <h5><?php echo number_format($stats['total']); ?></h5>
                                <small>Total Users</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2 col-6 mb-3">
                        <div class="card stat-card bg-success text-white">
                            <div class="card-body text-center py-3">
                                <i class="fas fa-user-check fa-lg mb-2"></i>
                                <h5><?php echo number_format($stats['active']); ?></h5>
                                <small>Active Users</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2 col-6 mb-3">
                        <div class="card stat-card bg-warning text-white">
                            <div class="card-body text-center py-3">
                                <i class="fas fa-crown fa-lg mb-2"></i>
                                <h5><?php echo number_format($stats['vip']); ?></h5>
                                <small>VIP Users</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2 col-6 mb-3">
                        <div class="card stat-card bg-danger text-white">
                            <div class="card-body text-center py-3">
                                <i class="fas fa-ban fa-lg mb-2"></i>
                                <h5><?php echo number_format($stats['banned']); ?></h5>
                                <small>Banned Users</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2 col-6 mb-3">
                        <div class="card stat-card bg-info text-white">
                            <div class="card-body text-center py-3">
                                <i class="fas fa-user-plus fa-lg mb-2"></i>
                                <h5><?php echo number_format($stats['today']); ?></h5>
                                <small>Today's Signups</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-2 col-6 mb-3">
                        <div class="card stat-card bg-secondary text-white">
                            <div class="card-body text-center py-3">
                                <i class="fas fa-chart-line fa-lg mb-2"></i>
                                <h5><?php echo number_format($totalUsers); ?></h5>
                                <small>Showing</small>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Filters and Search -->
                <div class="filter-box">
                    <form method="GET" class="row g-3">
                        <div class="col-md-4">
                            <input type="text" class="form-control" name="search" 
                                   placeholder="Search by User ID, Email, Phone..." 
                                   value="<?php echo htmlspecialchars($search); ?>">
                        </div>
                        <div class="col-md-2">
                            <select class="form-control" name="status">
                                <option value="">All Status</option>
                                <option value="active">Active</option>
                                <option value="banned">Banned</option>
                                <option value="inactive">Inactive</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <select class="form-control" name="vip">
                                <option value="">All Users</option>
                                <option value="1">VIP Only</option>
                                <option value="0">Non-VIP Only</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <input type="date" class="form-control" name="date" 
                                   placeholder="Registration Date">
                        </div>
                        <div class="col-md-2">
                            <button type="submit" class="btn btn-pink w-100">
                                <i class="fas fa-search me-1"></i> Search
                            </button>
                        </div>
                    </form>
                </div>
                
                <!-- Users Table -->
                <div class="card">
                    <div class="card-header bg-pink text-white d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><i class="fas fa-list me-2"></i> User List</h5>
                        <span>Page <?php echo $page; ?> of <?php echo $totalPages; ?></span>
                    </div>
                    <div class="card-body">
                        <?php if (empty($users)): ?>
                        <div class="text-center py-5">
                            <i class="fas fa-users-slash fa-3x text-muted mb-3"></i>
                            <p class="text-muted">No users found</p>
                            <?php if ($search): ?>
                            <a href="users.php" class="btn btn-pink">Clear Search</a>
                            <?php endif; ?>
                        </div>
                        <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>User ID</th>
                                        <th>Profile</th>
                                        <th>Country</th>
                                        <th>Status</th>
                                        <th>VIP</th>
                                        <th>Coins</th>
                                        <th>Registered</th>
                                        <th>Last Active</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($users as $user): ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo htmlspecialchars($user['user_id']); ?></strong>
                                            <?php if ($user['email']): ?>
                                            <br><small class="text-muted"><?php echo htmlspecialchars($user['email']); ?></small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($user['profile_image']): ?>
                                            <img src="<?php echo htmlspecialchars($user['profile_image']); ?>" 
                                                 class="user-avatar" alt="Profile">
                                            <?php else: ?>
                                            <div class="user-avatar bg-secondary d-flex align-items-center justify-content-center">
                                                <i class="fas fa-user text-white"></i>
                                            </div>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($user['country']): ?>
                                            <img src="assets/flags/<?php echo strtolower($user['country']); ?>.png" 
                                                 class="country-flag" alt="<?php echo $user['country']; ?>">
                                            <?php echo htmlspecialchars($user['country']); ?>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php 
                                                echo $user['status'] == 'active' ? 'success' : 
                                                     ($user['status'] == 'banned' ? 'danger' : 'warning');
                                            ?> status-badge">
                                                <?php echo ucfirst($user['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($user['is_vip']): ?>
                                            <span class="badge bg-warning status-badge">
                                                <i class="fas fa-crown me-1"></i>VIP
                                                <?php if ($user['vip_expiry']): ?>
                                                <br><small>Exp: <?php echo date('d/m/Y', strtotime($user['vip_expiry'])); ?></small>
                                                <?php endif; ?>
                                            </span>
                                            <?php else: ?>
                                            <span class="badge bg-secondary status-badge">Standard</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-info">
                                                <?php echo number_format($user['coins'] ?? 0); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php echo formatDate($user['created_at'], 'd/m/Y'); ?>
                                        </td>
                                        <td>
                                            <?php echo $user['last_active'] ? formatDate($user['last_active'], 'd/m/Y H:i') : 'Never'; ?>
                                        </td>
                                        <td>
                                            <div class="dropdown">
                                                <button class="btn btn-sm btn-outline-secondary dropdown-toggle" 
                                                        type="button" data-bs-toggle="dropdown">
                                                    <i class="fas fa-cog"></i>
                                                </button>
                                                <ul class="dropdown-menu action-dropdown">
                                                    <li>
                                                        <a class="dropdown-item" href="#" 
                                                           onclick="viewUserDetails('<?php echo $user['user_id']; ?>')">
                                                            <i class="fas fa-eye me-2"></i> View Details
                                                        </a>
                                                    </li>
                                                    <li><hr class="dropdown-divider"></li>
                                                    <?php if ($user['status'] == 'active'): ?>
                                                    <li>
                                                        <a class="dropdown-item text-danger" 
                                                           href="users.php?action=ban&id=<?php echo $user['user_id']; ?>">
                                                            <i class="fas fa-ban me-2"></i> Ban User
                                                        </a>
                                                    </li>
                                                    <?php else: ?>
                                                    <li>
                                                        <a class="dropdown-item text-success" 
                                                           href="users.php?action=unban&id=<?php echo $user['user_id']; ?>">
                                                            <i class="fas fa-check me-2"></i> Unban User
                                                        </a>
                                                    </li>
                                                    <?php endif; ?>
                                                    
                                                    <?php if (!$user['is_vip']): ?>
                                                    <li>
                                                        <a class="dropdown-item text-warning" 
                                                           href="users.php?action=make_vip&id=<?php echo $user['user_id']; ?>">
                                                            <i class="fas fa-crown me-2"></i> Make VIP
                                                        </a>
                                                    </li>
                                                    <?php else: ?>
                                                    <li>
                                                        <a class="dropdown-item text-secondary" 
                                                           href="users.php?action=remove_vip&id=<?php echo $user['user_id']; ?>">
                                                            <i class="fas fa-times me-2"></i> Remove VIP
                                                        </a>
                                                    </li>
                                                    <?php endif; ?>
                                                    
                                                    <li><hr class="dropdown-divider"></li>
                                                    <li>
                                                        <a class="dropdown-item text-danger" 
                                                           href="users.php?action=delete&id=<?php echo $user['user_id']; ?>"
                                                           onclick="return confirm('Are you sure you want to delete this user? This action cannot be undone.')">
                                                            <i class="fas fa-trash me-2"></i> Delete User
                                                        </a>
                                                    </li>
                                                </ul>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <!-- Pagination -->
                        <nav aria-label="Page navigation">
                            <ul class="pagination justify-content-center">
                                <?php if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=1<?php echo $search ? '&search=' . urlencode($search) : ''; ?>">
                                        <i class="fas fa-angle-double-left"></i>
                                    </a>
                                </li>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page - 1; ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>">
                                        <i class="fas fa-angle-left"></i>
                                    </a>
                                </li>
                                <?php endif; ?>
                                
                                <?php
                                $start = max(1, $page - 2);
                                $end = min($totalPages, $page + 2);
                                
                                for ($i = $start; $i <= $end; $i++):
                                ?>
                                <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $i; ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>">
                                        <?php echo $i; ?>
                                    </a>
                                </li>
                                <?php endfor; ?>
                                
                                <?php if ($page < $totalPages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page + 1; ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>">
                                        <i class="fas fa-angle-right"></i>
                                    </a>
                                </li>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $totalPages; ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>">
                                        <i class="fas fa-angle-double-right"></i>
                                    </a>
                                </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- User Details Modal -->
    <div class="modal fade" id="userDetailsModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-pink text-white">
                    <h5 class="modal-title">User Details</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="userDetailsContent">
                        <!-- Content will be loaded via AJAX -->
                        <div class="text-center py-4">
                            <div class="spinner-border text-pink" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                            <p class="mt-2">Loading user details...</p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/scripts.php'; ?>
    <script>
    function viewUserDetails(userId) {
        // Load user details via AJAX
        $('#userDetailsContent').html(`
            <div class="text-center py-4">
                <div class="spinner-border text-pink" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="mt-2">Loading user details...</p>
            </div>
        `);
        
        $.get('ajax/get_user_details.php', {user_id: userId}, function(response) {
            if (response.success) {
                $('#userDetailsContent').html(response.html);
            } else {
                $('#userDetailsContent').html(`
                    <div class="alert alert-danger">
                        Error loading user details: ${response.message}
                    </div>
                `);
            }
        }, 'json');
        
        var modal = new bootstrap.Modal(document.getElementById('userDetailsModal'));
        modal.show();
    }
    
    function exportUsers() {
        var format = prompt('Export format (csv/json/excel):', 'csv');
        if (format && ['csv', 'json', 'excel'].includes(format.toLowerCase())) {
            window.location.href = 'ajax/export_users.php?format=' + format + '&search=<?php echo urlencode($search); ?>';
        }
    }
    
    // Quick status filter
    document.querySelectorAll('.quick-filter').forEach(btn => {
        btn.addEventListener('click', function() {
            var status = this.dataset.status;
            window.location.href = 'users.php?status=' + status;
        });
    });
    </script>
</body>
</html>