<?php
// updates.php - App Updates Management
session_start();
require_once 'includes/config.php';
require_once 'includes/functions.php';

checkAdminLogin();

$message = '';
$error = '';

// Handle APK upload
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_FILES['apk_file'])) {
    $versionCode = intval($_POST['version_code'] ?? 0);
    $versionName = sanitize_input($_POST['version_name'] ?? '');
    $changelog = sanitize_input($_POST['changelog'] ?? '');
    $forceUpdate = isset($_POST['force_update']) ? 1 : 0;
    
    if ($versionCode < 1 || empty($versionName) || empty($changelog)) {
        $error = 'Please fill all required fields';
    } elseif ($_FILES['apk_file']['error'] !== UPLOAD_ERR_OK) {
        $error = 'File upload error: ' . $_FILES['apk_file']['error'];
    } else {
        // Validate file
        $allowedTypes = ['apk', 'ipa'];
        $extension = strtolower(pathinfo($_FILES['apk_file']['name'], PATHINFO_EXTENSION));
        
        if (!in_array($extension, $allowedTypes)) {
            $error = 'Only APK or IPA files are allowed';
        } elseif ($_FILES['apk_file']['size'] > 100 * 1024 * 1024) { // 100MB max
            $error = 'File too large. Maximum size is 100MB';
        } else {
            // Create upload directory
            $uploadDir = 'uploads/apks/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0777, true);
            }
            
            // Generate filename
            $filename = 'pink_video_chat_v' . $versionCode . '_' . time() . '.' . $extension;
            $filepath = $uploadDir . $filename;
            
            if (move_uploaded_file($_FILES['apk_file']['tmp_name'], $filepath)) {
                // Generate download URL
                $apkUrl = (isset($_SERVER['HTTPS']) ? 'https://' : 'http://') . 
                         $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . '/' . $filepath;
                
                // Save to database
                if (addAppUpdate($versionCode, $versionName, $apkUrl, $changelog, $forceUpdate)) {
                    $message = 'APK uploaded successfully! Update is now live.';
                    
                    // Send push notification about update
                    sendPushNotification(
                        'New Update Available!',
                        'Version ' . $versionName . ' is now available. ' . 
                        ($forceUpdate ? 'This is a required update.' : ''),
                        'all'
                    );
                } else {
                    $error = 'Failed to save update information';
                    unlink($filepath); // Remove uploaded file
                }
            } else {
                $error = 'Failed to upload file';
            }
        }
    }
}

// Handle update actions
if (isset($_GET['action'])) {
    $updateId = intval($_GET['id'] ?? 0);
    $action = $_GET['action'];
    
    $db = getDB();
    
    switch ($action) {
        case 'activate':
            $stmt = $db->prepare("UPDATE app_updates SET is_active = 1 WHERE id = ?");
            if ($stmt->execute([$updateId])) {
                $message = 'Update activated successfully';
            } else {
                $error = 'Failed to activate update';
            }
            break;
            
        case 'deactivate':
            $stmt = $db->prepare("UPDATE app_updates SET is_active = 0 WHERE id = ?");
            if ($stmt->execute([$updateId])) {
                $message = 'Update deactivated successfully';
            } else {
                $error = 'Failed to deactivate update';
            }
            break;
            
        case 'delete':
            // Get file path before deleting
            $stmt = $db->prepare("SELECT apk_url FROM app_updates WHERE id = ?");
            $stmt->execute([$updateId]);
            $update = $stmt->fetch();
            
            // Delete from database
            $stmt = $db->prepare("DELETE FROM app_updates WHERE id = ?");
            if ($stmt->execute([$updateId])) {
                // Delete file
                if ($update && file_exists(str_replace((isset($_SERVER['HTTPS']) ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'], '.', $update['apk_url']))) {
                    unlink(str_replace((isset($_SERVER['HTTPS']) ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'], '.', $update['apk_url']));
                }
                $message = 'Update deleted successfully';
            } else {
                $error = 'Failed to delete update';
            }
            break;
            
        case 'force':
            $force = $_GET['force'] == '1' ? 1 : 0;
            $stmt = $db->prepare("UPDATE app_updates SET force_update = ? WHERE id = ?");
            if ($stmt->execute([$force, $updateId])) {
                $message = 'Force update ' . ($force ? 'enabled' : 'disabled');
            } else {
                $error = 'Failed to update settings';
            }
            break;
    }
}

// Get all updates
$db = getDB();
$stmt = $db->query("SELECT * FROM app_updates ORDER BY version_code DESC");
$updates = $stmt->fetchAll();

// Get current active version
$currentVersion = getCurrentVersion();
function getCurrentVersion() {
    $db = getDB();
    $stmt = $db->query("SELECT * FROM app_updates WHERE is_active = 1 ORDER BY version_code DESC LIMIT 1");
    return $stmt->fetch();
}

// Get update statistics
$stats = [
    'total_updates' => count($updates),
    'active_updates' => count(array_filter($updates, fn($u) => $u['is_active'])),
    'force_updates' => count(array_filter($updates, fn($u) => $u['force_update'])),
    'total_downloads' => getTotalDownloads()
];

function getTotalDownloads() {
    $db = getDB();
    $stmt = $db->query("SELECT SUM(download_count) as total FROM app_updates");
    $result = $stmt->fetch();
    return $result['total'] ?? 0;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>App Updates - Pink Video Chat Admin</title>
    <?php include 'includes/head.php'; ?>
    <style>
        .update-card {
            border-left: 4px solid #FF69B4;
            margin-bottom: 20px;
            transition: all 0.3s;
        }
        .update-card.active {
            border-left-color: #28a745;
            box-shadow: 0 5px 15px rgba(40, 167, 69, 0.2);
        }
        .update-card.inactive {
            border-left-color: #6c757d;
            opacity: 0.8;
        }
        .update-card.force {
            border-left-color: #dc3545;
            box-shadow: 0 5px 15px rgba(220, 53, 69, 0.2);
        }
        .version-badge {
            font-size: 1rem;
            padding: 5px 15px;
        }
        .download-btn {
            position: relative;
            overflow: hidden;
        }
        .download-btn .badge {
            position: absolute;
            top: -5px;
            right: -5px;
            font-size: 0.6rem;
        }
        .changelog {
            background: #f8f9fa;
            border-radius: 5px;
            padding: 15px;
            margin-top: 10px;
            max-height: 200px;
            overflow-y: auto;
        }
        .file-upload-box {
            border: 2px dashed #dee2e6;
            border-radius: 10px;
            padding: 40px 20px;
            text-align: center;
            background: #f8f9fa;
            cursor: pointer;
            transition: all 0.3s;
        }
        .file-upload-box:hover {
            border-color: #FF69B4;
            background: rgba(255, 105, 180, 0.05);
        }
        .file-upload-box.dragover {
            border-color: #28a745;
            background: rgba(40, 167, 69, 0.1);
        }
        .progress {
            height: 25px;
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container mt-4">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="col-md-9">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h3><i class="fas fa-sync-alt me-2"></i> App Updates Management</h3>
                    <button class="btn btn-pink" data-bs-toggle="modal" data-bs-target="#uploadModal">
                        <i class="fas fa-upload me-1"></i> Upload New APK
                    </button>
                </div>
                
                <?php if ($message): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <?php echo $message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <?php echo $error; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <!-- Update Statistics -->
                <div class="row mb-4">
                    <div class="col-md-3 col-6 mb-3">
                        <div class="card stat-card bg-primary text-white">
                            <div class="card-body text-center py-3">
                                <i class="fas fa-code-branch fa-lg mb-2"></i>
                                <h5><?php echo $stats['total_updates']; ?></h5>
                                <small>Total Versions</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 col-6 mb-3">
                        <div class="card stat-card bg-success text-white">
                            <div class="card-body text-center py-3">
                                <i class="fas fa-check-circle fa-lg mb-2"></i>
                                <h5><?php echo $stats['active_updates']; ?></h5>
                                <small>Active Versions</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 col-6 mb-3">
                        <div class="card stat-card bg-danger text-white">
                            <div class="card-body text-center py-3">
                                <i class="fas fa-exclamation-circle fa-lg mb-2"></i>
                                <h5><?php echo $stats['force_updates']; ?></h5>
                                <small>Force Updates</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 col-6 mb-3">
                        <div class="card stat-card bg-info text-white">
                            <div class="card-body text-center py-3">
                                <i class="fas fa-download fa-lg mb-2"></i>
                                <h5><?php echo number_format($stats['total_downloads']); ?></h5>
                                <small>Total Downloads</small>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Current Version Info -->
                <?php if ($currentVersion): ?>
                <div class="card mb-4 update-card active">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start">
                            <div>
                                <h4 class="card-title mb-1">
                                    <span class="badge bg-success version-badge me-2">
                                        v<?php echo $currentVersion['version_name']; ?>
                                    </span>
                                    Current Active Version
                                </h4>
                                <p class="text-muted mb-2">
                                    <i class="fas fa-calendar me-1"></i> 
                                    Released: <?php echo formatDate($currentVersion['release_date'], 'F j, Y'); ?>
                                </p>
                                <?php if ($currentVersion['force_update']): ?>
                                <span class="badge bg-danger me-2">
                                    <i class="fas fa-exclamation-triangle me-1"></i> Force Update Required
                                </span>
                                <?php endif; ?>
                                <span class="badge bg-secondary">
                                    <i class="fas fa-download me-1"></i> 
                                    <?php echo number_format($currentVersion['download_count'] ?? 0); ?> downloads
                                </span>
                            </div>
                            <div>
                                <a href="<?php echo $currentVersion['apk_url']; ?>" 
                                   class="btn btn-pink download-btn" download>
                                    <i class="fas fa-download me-1"></i> Download APK
                                    <?php if ($currentVersion['download_count'] > 0): ?>
                                    <span class="badge bg-light text-dark">
                                        <?php echo $currentVersion['download_count']; ?>
                                    </span>
                                    <?php endif; ?>
                                </a>
                            </div>
                        </div>
                        <div class="mt-3">
                            <h6><i class="fas fa-clipboard-list me-2"></i> Changelog:</h6>
                            <div class="changelog">
                                <?php echo nl2br(htmlspecialchars($currentVersion['changelog'])); ?>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- All Updates -->
                <div class="card">
                    <div class="card-header bg-pink text-white d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><i class="fas fa-history me-2"></i> All App Versions</h5>
                        <span><?php echo count($updates); ?> versions total</span>
                    </div>
                    <div class="card-body">
                        <?php if (empty($updates)): ?>
                        <div class="text-center py-5">
                            <i class="fas fa-box-open fa-3x text-muted mb-3"></i>
                            <p class="text-muted">No app versions uploaded yet</p>
                            <button class="btn btn-pink" data-bs-toggle="modal" data-bs-target="#uploadModal">
                                <i class="fas fa-upload me-1"></i> Upload First Version
                            </button>
                        </div>
                        <?php else: ?>
                        <div class="row">
                            <?php foreach ($updates as $update): ?>
                            <div class="col-md-6 mb-3">
                                <div class="card update-card <?php 
                                    echo $update['is_active'] ? 'active' : 'inactive';
                                    echo $update['force_update'] ? ' force' : '';
                                ?>">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between align-items-start mb-2">
                                            <div>
                                                <h5 class="card-title mb-1">
                                                    <span class="badge bg-<?php 
                                                        echo $update['is_active'] ? 'success' : 'secondary';
                                                    ?> me-2">
                                                        v<?php echo $update['version_name']; ?>
                                                    </span>
                                                    (Code: <?php echo $update['version_code']; ?>)
                                                </h5>
                                                <p class="text-muted mb-1">
                                                    <i class="fas fa-calendar me-1"></i> 
                                                    <?php echo formatDate($update['release_date'], 'M d, Y'); ?>
                                                </p>
                                            </div>
                                            <div class="btn-group">
                                                <button type="button" class="btn btn-sm btn-outline-secondary dropdown-toggle" 
                                                        data-bs-toggle="dropdown">
                                                    <i class="fas fa-cog"></i>
                                                </button>
                                                <ul class="dropdown-menu">
                                                    <?php if (!$update['is_active']): ?>
                                                    <li>
                                                        <a class="dropdown-item text-success" 
                                                           href="updates.php?action=activate&id=<?php echo $update['id']; ?>">
                                                            <i class="fas fa-check me-2"></i> Activate
                                                        </a>
                                                    </li>
                                                    <?php else: ?>
                                                    <li>
                                                        <a class="dropdown-item text-warning" 
                                                           href="updates.php?action=deactivate&id=<?php echo $update['id']; ?>">
                                                            <i class="fas fa-times me-2"></i> Deactivate
                                                        </a>
                                                    </li>
                                                    <?php endif; ?>
                                                    
                                                    <?php if (!$update['force_update']): ?>
                                                    <li>
                                                        <a class="dropdown-item text-danger" 
                                                           href="updates.php?action=force&id=<?php echo $update['id']; ?>&force=1">
                                                            <i class="fas fa-exclamation-triangle me-2"></i> Make Force Update
                                                        </a>
                                                    </li>
                                                    <?php else: ?>
                                                    <li>
                                                        <a class="dropdown-item text-info" 
                                                           href="updates.php?action=force&id=<?php echo $update['id']; ?>&force=0">
                                                            <i class="fas fa-check-circle me-2"></i> Remove Force Update
                                                        </a>
                                                    </li>
                                                    <?php endif; ?>
                                                    
                                                    <li><hr class="dropdown-divider"></li>
                                                    <li>
                                                        <a class="dropdown-item text-danger" 
                                                           href="updates.php?action=delete&id=<?php echo $update['id']; ?>"
                                                           onclick="return confirm('Are you sure? This will delete the APK file and update record.')">
                                                            <i class="fas fa-trash me-2"></i> Delete
                                                        </a>
                                                    </li>
                                                </ul>
                                            </div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <span class="badge bg-<?php echo $update['is_active'] ? 'success' : 'secondary'; ?> me-2">
                                                <?php echo $update['is_active'] ? 'Active' : 'Inactive'; ?>
                                            </span>
                                            <?php if ($update['force_update']): ?>
                                            <span class="badge bg-danger me-2">
                                                <i class="fas fa-exclamation-triangle me-1"></i> Force Update
                                            </span>
                                            <?php endif; ?>
                                            <span class="badge bg-info">
                                                <i class="fas fa-download me-1"></i> 
                                                <?php echo number_format($update['download_count'] ?? 0); ?> downloads
                                            </span>
                                        </div>
                                        
                                        <div class="changelog mb-3">
                                            <strong>Changelog:</strong><br>
                                            <?php echo nl2br(htmlspecialchars(substr($update['changelog'], 0, 200))); ?>
                                            <?php if (strlen($update['changelog']) > 200): ?>...<?php endif; ?>
                                        </div>
                                        
                                        <div class="d-flex justify-content-between align-items-center">
                                            <a href="<?php echo $update['apk_url']; ?>" 
                                               class="btn btn-sm btn-outline-primary" download>
                                                <i class="fas fa-download me-1"></i> Download
                                            </a>
                                            <small class="text-muted">
                                                <i class="fas fa-link me-1"></i>
                                                <a href="<?php echo $update['apk_url']; ?>" target="_blank" class="text-decoration-none">
                                                    Direct Link
                                                </a>
                                            </small>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Upload Modal -->
    <div class="modal fade" id="uploadModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-pink text-white">
                    <h5 class="modal-title"><i class="fas fa-upload me-2"></i> Upload New APK Version</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form method="POST" enctype="multipart/form-data" id="uploadForm">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Version Code *</label>
                                <input type="number" class="form-control" name="version_code" 
                                       placeholder="e.g., 2" min="1" required>
                                <small class="text-muted">Must be higher than current version (<?php 
                                    echo $currentVersion ? $currentVersion['version_code'] : 0; ?>)
                                </small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Version Name *</label>
                                <input type="text" class="form-control" name="version_name" 
                                       placeholder="e.g., 1.2.0" required>
                                <small class="text-muted">Display name for users</small>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Changelog *</label>
                            <textarea class="form-control" name="changelog" rows="4" 
                                      placeholder="List the changes in this version..." required></textarea>
                            <small class="text-muted">What's new in this update?</small>
                        </div>
                        
                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="force_update" id="forceUpdate">
                                <label class="form-check-label" for="forceUpdate">
                                    <strong>Force Update</strong> - Users must update to continue using the app
                                </label>
                            </div>
                            <small class="text-muted">Use this for critical updates only</small>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">APK File *</label>
                            <div class="file-upload-box" id="fileDropZone">
                                <i class="fas fa-cloud-upload-alt fa-3x text-muted mb-3"></i>
                                <p class="mb-2">Drag & drop APK file here or click to browse</p>
                                <p class="text-muted small mb-3">Maximum file size: 100MB</p>
                                <input type="file" class="form-control d-none" name="apk_file" id="apkFile" accept=".apk,.ipa" required>
                                <button type="button" class="btn btn-outline-secondary" onclick="document.getElementById('apkFile').click()">
                                    <i class="fas fa-folder-open me-2"></i> Browse Files
                                </button>
                                <div class="mt-3" id="fileInfo"></div>
                            </div>
                        </div>
                        
                        <div class="progress d-none" id="uploadProgress">
                            <div class="progress-bar progress-bar-striped progress-bar-animated" 
                                 role="progressbar" style="width: 0%"></div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-pink" onclick="uploadAPK()" id="uploadBtn">
                        <i class="fas fa-upload me-2"></i> Upload & Publish
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/scripts.php'; ?>
    <script>
    // File upload handling
    const fileInput = document.getElementById('apkFile');
    const fileDropZone = document.getElementById('fileDropZone');
    const fileInfo = document.getElementById('fileInfo');
    
    fileInput.addEventListener('change', function(e) {
        handleFiles(this.files);
    });
    
    // Drag and drop
    fileDropZone.addEventListener('dragover', function(e) {
        e.preventDefault();
        this.classList.add('dragover');
    });
    
    fileDropZone.addEventListener('dragleave', function(e) {
        this.classList.remove('dragover');
    });
    
    fileDropZone.addEventListener('drop', function(e) {
        e.preventDefault();
        this.classList.remove('dragover');
        handleFiles(e.dataTransfer.files);
    });
    
    function handleFiles(files) {
        if (files.length > 0) {
            const file = files[0];
            const fileSize = (file.size / (1024 * 1024)).toFixed(2);
            
            if (!file.name.match(/\.(apk|ipa)$/i)) {
                alert('Only APK or IPA files are allowed');
                return;
            }
            
            if (file.size > 100 * 1024 * 1024) {
                alert('File too large. Maximum size is 100MB');
                return;
            }
            
            fileInput.files = files;
            
            fileInfo.innerHTML = `
                <div class="alert alert-success">
                    <i class="fas fa-check-circle me-2"></i>
                    <strong>${file.name}</strong> (${fileSize} MB)
                </div>
            `;
        }
    }
    
    function uploadAPK() {
        const form = document.getElementById('uploadForm');
        const formData = new FormData(form);
        
        // Validate version code
        const versionCode = parseInt(formData.get('version_code'));
        const currentVersion = <?php echo $currentVersion ? $currentVersion['version_code'] : 0; ?>;
        
        if (versionCode <= currentVersion) {
            alert(`Version code must be greater than current version (${currentVersion})`);
            return;
        }
        
        // Show progress bar
        const progressBar = document.getElementById('uploadProgress');
        const uploadBtn = document.getElementById('uploadBtn');
        
        progressBar.classList.remove('d-none');
        uploadBtn.disabled = true;
        uploadBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i> Uploading...';
        
        // Simulate upload progress (in real app, use XMLHttpRequest with progress event)
        let progress = 0;
        const progressInterval = setInterval(() => {
            progress += 10;
            progressBar.querySelector('.progress-bar').style.width = progress + '%';
            
            if (progress >= 90) {
                clearInterval(progressInterval);
                // Submit form
                form.submit();
            }
        }, 200);
    }
    
    // Auto-suggest version name based on code
    document.querySelector('input[name="version_code"]').addEventListener('input', function() {
        const versionNameInput = document.querySelector('input[name="version_name"]');
        if (!versionNameInput.value && this.value) {
            const major = Math.floor(this.value / 100);
            const minor = Math.floor((this.value % 100) / 10);
            const patch = this.value % 10;
            versionNameInput.value = `${major}.${minor}.${patch}`;
        }
    });
    
    // Format changelog on paste
    document.querySelector('textarea[name="changelog"]').addEventListener('paste', function(e) {
        setTimeout(() => {
            let text = this.value;
            // Convert bullet points
            text = text.replace(/•/g, '•');
            // Add line breaks
            text = text.replace(/\n/g, '\n• ');
            this.value = '• ' + text;
        }, 10);
    });
    </script>
</body>
</html>