<?php
// revenue.php - Revenue Management
session_start();
require_once 'includes/config.php';
require_once 'includes/functions.php';

checkAdminLogin();

// Get time period
$period = isset($_GET['period']) ? sanitize_input($_GET['period']) : 'today';
$validPeriods = ['today', 'week', 'month', 'year', 'all'];
if (!in_array($period, $validPeriods)) {
    $period = 'today';
}

// Get revenue statistics
$revenueToday = getRevenueStats('today');
$revenueWeek = getRevenueStats('week');
$revenueMonth = getRevenueStats('month');
$revenueYear = getRevenueStats('year');
$revenueAll = getRevenueStats('all');

// Get recent transactions
$recentTransactions = getRecentTransactions(20);

// Get transaction types breakdown
$transactionTypes = getTransactionTypes();
function getTransactionTypes() {
    $db = getDB();
    $stmt = $db->query("SELECT type, COUNT(*) as count, SUM(amount) as total 
                       FROM revenue_transactions 
                       WHERE status = 'completed' 
                       GROUP BY type ORDER BY total DESC");
    return $stmt->fetchAll();
}

// Get daily revenue for chart
$dailyRevenue = getDailyRevenue(30);
function getDailyRevenue($days = 30) {
    $db = getDB();
    $stmt = $db->prepare("SELECT DATE(created_at) as date, SUM(amount) as total 
                         FROM revenue_transactions 
                         WHERE status = 'completed' 
                         AND created_at >= DATE_SUB(CURDATE(), INTERVAL ? DAY)
                         GROUP BY DATE(created_at) 
                         ORDER BY date DESC");
    $stmt->execute([$days]);
    return $stmt->fetchAll();
}

// Handle refund
if (isset($_GET['refund'])) {
    $transactionId = $_GET['refund'];
    $db = getDB();
    
    $stmt = $db->prepare("UPDATE revenue_transactions SET status = 'refunded' WHERE transaction_id = ?");
    if ($stmt->execute([$transactionId])) {
        $message = 'Transaction refunded successfully';
    } else {
        $error = 'Failed to refund transaction';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Revenue Management - Pink Video Chat Admin</title>
    <?php include 'includes/head.php'; ?>
    <style>
        .revenue-card {
            border-radius: 10px;
            color: white;
            padding: 20px;
            margin-bottom: 20px;
            text-align: center;
        }
        .revenue-card h3 {
            font-size: 1.8rem;
            margin-bottom: 5px;
            font-weight: bold;
        }
        .revenue-card p {
            margin-bottom: 0;
            opacity: 0.9;
        }
        .revenue-card i {
            font-size: 2.5rem;
            margin-bottom: 15px;
            opacity: 0.8;
        }
        .bg-today { background: linear-gradient(45deg, #f093fb, #f5576c); }
        .bg-week { background: linear-gradient(45deg, #4facfe, #00f2fe); }
        .bg-month { background: linear-gradient(45deg, #43e97b, #38f9d7); }
        .bg-year { background: linear-gradient(45deg, #fa709a, #fee140); }
        .bg-all { background: linear-gradient(45deg, #667eea, #764ba2); }
        
        .type-badge {
            font-size: 0.75rem;
            padding: 3px 10px;
        }
        .chart-container {
            position: relative;
            height: 300px;
            margin-bottom: 30px;
        }
        .period-nav .nav-link {
            color: #666;
            font-weight: 500;
        }
        .period-nav .nav-link.active {
            color: #FF1493;
            border-bottom: 3px solid #FF1493;
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container mt-4">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="col-md-9">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h3><i class="fas fa-dollar-sign me-2"></i> Revenue Management</h3>
                    <div>
                        <button class="btn btn-pink" onclick="exportRevenue()">
                            <i class="fas fa-download me-1"></i> Export Report
                        </button>
                    </div>
                </div>
                
                <!-- Revenue Statistics Cards -->
                <div class="row mb-4">
                    <div class="col-md-2 col-6 mb-3">
                        <div class="revenue-card bg-today">
                            <i class="fas fa-calendar-day"></i>
                            <h3>$<?php echo number_format($revenueToday, 2); ?></h3>
                            <p>Today</p>
                        </div>
                    </div>
                    <div class="col-md-2 col-6 mb-3">
                        <div class="revenue-card bg-week">
                            <i class="fas fa-calendar-week"></i>
                            <h3>$<?php echo number_format($revenueWeek, 2); ?></h3>
                            <p>This Week</p>
                        </div>
                    </div>
                    <div class="col-md-2 col-6 mb-3">
                        <div class="revenue-card bg-month">
                            <i class="fas fa-calendar-alt"></i>
                            <h3>$<?php echo number_format($revenueMonth, 2); ?></h3>
                            <p>This Month</p>
                        </div>
                    </div>
                    <div class="col-md-2 col-6 mb-3">
                        <div class="revenue-card bg-year">
                            <i class="fas fa-calendar"></i>
                            <h3>$<?php echo number_format($revenueYear, 2); ?></h3>
                            <p>This Year</p>
                        </div>
                    </div>
                    <div class="col-md-2 col-6 mb-3">
                        <div class="revenue-card bg-all">
                            <i class="fas fa-chart-line"></i>
                            <h3>$<?php echo number_format($revenueAll, 2); ?></h3>
                            <p>All Time</p>
                        </div>
                    </div>
                    <div class="col-md-2 col-6 mb-3">
                        <div class="revenue-card bg-secondary">
                            <i class="fas fa-exchange-alt"></i>
                            <h3><?php echo count($recentTransactions); ?></h3>
                            <p>Transactions</p>
                        </div>
                    </div>
                </div>
                
                <!-- Period Navigation -->
                <div class="card mb-4">
                    <div class="card-body">
                        <ul class="nav nav-pills period-nav">
                            <li class="nav-item">
                                <a class="nav-link <?php echo $period == 'today' ? 'active' : ''; ?>" 
                                   href="?period=today">Today</a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link <?php echo $period == 'week' ? 'active' : ''; ?>" 
                                   href="?period=week">This Week</a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link <?php echo $period == 'month' ? 'active' : ''; ?>" 
                                   href="?period=month">This Month</a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link <?php echo $period == 'year' ? 'active' : ''; ?>" 
                                   href="?period=year">This Year</a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link <?php echo $period == 'all' ? 'active' : ''; ?>" 
                                   href="?period=all">All Time</a>
                            </li>
                        </ul>
                    </div>
                </div>
                
                <!-- Revenue Chart -->
                <div class="card mb-4">
                    <div class="card-header bg-pink text-white">
                        <h5 class="mb-0"><i class="fas fa-chart-line me-2"></i> Revenue Chart (Last 30 Days)</h5>
                    </div>
                    <div class="card-body">
                        <div class="chart-container">
                            <canvas id="revenueChart"></canvas>
                        </div>
                    </div>
                </div>
                
                <!-- Transaction Types -->
                <div class="row mb-4">
                    <div class="col-md-6">
                        <div class="card h-100">
                            <div class="card-header bg-pink text-white">
                                <h5 class="mb-0"><i class="fas fa-chart-pie me-2"></i> Transaction Types</h5>
                            </div>
                            <div class="card-body">
                                <div class="chart-container">
                                    <canvas id="typeChart"></canvas>
                                </div>
                                <div class="mt-3">
                                    <?php foreach ($transactionTypes as $type): ?>
                                    <div class="d-flex justify-content-between align-items-center mb-2">
                                        <span>
                                            <span class="badge bg-info type-badge me-2">
                                                <?php echo ucfirst($type['type']); ?>
                                            </span>
                                            <?php echo $type['count']; ?> transactions
                                        </span>
                                        <strong>$<?php echo number_format($type['total'], 2); ?></strong>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-6">
                        <div class="card h-100">
                            <div class="card-header bg-pink text-white">
                                <h5 class="mb-0"><i class="fas fa-list me-2"></i> Quick Stats</h5>
                            </div>
                            <div class="card-body">
                                <?php
                                $stats = [
                                    'Average Transaction' => $revenueAll > 0 ? $revenueAll / array_sum(array_column($transactionTypes, 'count')) : 0,
                                    'VIP Conversions' => getVIPConversions(),
                                    'Refund Rate' => getRefundRate(),
                                    'Success Rate' => getSuccessRate(),
                                    'Peak Hours' => getPeakHours(),
                                    'Top Country' => getTopCountry()
                                ];
                                
                                foreach ($stats as $label => $value):
                                ?>
                                <div class="d-flex justify-content-between align-items-center mb-3 p-2 border-bottom">
                                    <span><?php echo $label; ?></span>
                                    <strong class="<?php echo strpos($label, 'Rate') !== false ? ($value < 5 ? 'text-success' : 'text-danger') : ''; ?>">
                                        <?php echo is_numeric($value) ? (strpos($label, 'Rate') !== false ? $value . '%' : '$' . number_format($value, 2)) : $value; ?>
                                    </strong>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Recent Transactions -->
                <div class="card">
                    <div class="card-header bg-pink text-white">
                        <h5 class="mb-0"><i class="fas fa-history me-2"></i> Recent Transactions</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($recentTransactions)): ?>
                        <div class="text-center py-5">
                            <i class="fas fa-exchange-alt fa-3x text-muted mb-3"></i>
                            <p class="text-muted">No transactions found</p>
                        </div>
                        <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Transaction ID</th>
                                        <th>User</th>
                                        <th>Type</th>
                                        <th>Amount</th>
                                        <th>Status</th>
                                        <th>Date</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recentTransactions as $transaction): ?>
                                    <tr>
                                        <td>
                                            <small class="text-muted"><?php echo substr($transaction['transaction_id'], 0, 8) . '...'; ?></small>
                                        </td>
                                        <td>
                                            <?php if ($transaction['user_id']): ?>
                                            <a href="users.php?search=<?php echo urlencode($transaction['user_id']); ?>" 
                                               class="text-decoration-none">
                                                <?php echo $transaction['user_id']; ?>
                                            </a>
                                            <?php else: ?>
                                            <span class="text-muted">Guest</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php 
                                                echo $transaction['type'] == 'subscription' ? 'success' : 
                                                     ($transaction['type'] == 'coins' ? 'warning' : 'info');
                                            ?> type-badge">
                                                <?php echo ucfirst($transaction['type']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <strong class="<?php echo $transaction['amount'] > 0 ? 'text-success' : 'text-danger'; ?>">
                                                $<?php echo number_format($transaction['amount'], 2); ?>
                                            </strong>
                                            <br>
                                            <small class="text-muted"><?php echo $transaction['currency']; ?></small>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php 
                                                echo $transaction['status'] == 'completed' ? 'success' : 
                                                     ($transaction['status'] == 'pending' ? 'warning' : 
                                                     ($transaction['status'] == 'refunded' ? 'danger' : 'secondary'));
                                            ?>">
                                                <?php echo ucfirst($transaction['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php echo formatDate($transaction['created_at'], 'd/m/Y H:i'); ?>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <button class="btn btn-outline-info" 
                                                        onclick="viewTransaction('<?php echo $transaction['transaction_id']; ?>')"
                                                        title="View Details">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                <?php if ($transaction['status'] == 'completed'): ?>
                                                <button class="btn btn-outline-danger" 
                                                        onclick="refundTransaction('<?php echo $transaction['transaction_id']; ?>')"
                                                        title="Refund">
                                                    <i class="fas fa-undo"></i>
                                                </button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <div class="text-center mt-3">
                            <a href="transactions.php" class="btn btn-pink">
                                <i class="fas fa-list me-1"></i> View All Transactions
                            </a>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Transaction Details Modal -->
    <div class="modal fade" id="transactionModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-pink text-white">
                    <h5 class="modal-title">Transaction Details</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="transactionDetails">
                        <!-- Content will be loaded via AJAX -->
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/scripts.php'; ?>
    <script>
    // Revenue Chart
    var revenueCtx = document.getElementById('revenueChart').getContext('2d');
    var revenueChart = new Chart(revenueCtx, {
        type: 'line',
        data: {
            labels: [<?php 
                $labels = [];
                $data = [];
                foreach ($dailyRevenue as $rev) {
                    $labels[] = "'" . date('d M', strtotime($rev['date'])) . "'";
                    $data[] = $rev['total'];
                }
                echo implode(',', array_reverse($labels));
            ?>],
            datasets: [{
                label: 'Daily Revenue ($)',
                data: [<?php echo implode(',', array_reverse($data)); ?>],
                borderColor: '#FF69B4',
                backgroundColor: 'rgba(255, 105, 180, 0.1)',
                fill: true,
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: true,
                    position: 'top'
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return '$' + value;
                        }
                    }
                }
            }
        }
    });
    
    // Type Chart
    var typeCtx = document.getElementById('typeChart').getContext('2d');
    var typeChart = new Chart(typeCtx, {
        type: 'doughnut',
        data: {
            labels: [<?php 
                $typeLabels = [];
                $typeData = [];
                $typeColors = ['#FF69B4', '#36A2EB', '#FFCE56', '#4BC0C0', '#9966FF'];
                $i = 0;
                foreach ($transactionTypes as $type) {
                    $typeLabels[] = "'" . ucfirst($type['type']) . "'";
                    $typeData[] = $type['total'];
                    $i++;
                }
                echo implode(',', $typeLabels);
            ?>],
            datasets: [{
                data: [<?php echo implode(',', $typeData); ?>],
                backgroundColor: [<?php 
                    $colors = [];
                    for ($i = 0; $i < count($transactionTypes); $i++) {
                        $colors[] = "'" . $typeColors[$i % count($typeColors)] . "'";
                    }
                    echo implode(',', $colors);
                ?>]
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'right'
                }
            }
        }
    });
    
    function viewTransaction(transactionId) {
        $.get('ajax/get_transaction.php', {id: transactionId}, function(response) {
            if (response.success) {
                $('#transactionDetails').html(response.html);
            } else {
                $('#transactionDetails').html('<div class="alert alert-danger">' + response.message + '</div>');
            }
        }, 'json');
        
        var modal = new bootstrap.Modal(document.getElementById('transactionModal'));
        modal.show();
    }
    
    function refundTransaction(transactionId) {
        if (confirm('Are you sure you want to refund this transaction?')) {
            window.location.href = 'revenue.php?refund=' + transactionId;
        }
    }
    
    function exportRevenue() {
        var format = prompt('Export format (csv/json/excel):', 'csv');
        if (format && ['csv', 'json', 'excel'].includes(format.toLowerCase())) {
            window.location.href = 'ajax/export_revenue.php?format=' + format + '&period=<?php echo $period; ?>';
        }
    }
    </script>
</body>
</html>

<?php
// Helper functions
function getVIPConversions() {
    $db = getDB();
    $stmt = $db->query("SELECT COUNT(DISTINCT user_id) as count FROM revenue_transactions 
                       WHERE type = 'subscription' AND status = 'completed'");
    $result = $stmt->fetch();
    return $result['count'] ?? 0;
}

function getRefundRate() {
    $db = getDB();
    $stmt = $db->query("SELECT 
        (SELECT COUNT(*) FROM revenue_transactions WHERE status = 'refunded') as refunded,
        (SELECT COUNT(*) FROM revenue_transactions WHERE status = 'completed') as completed");
    $result = $stmt->fetch();
    if ($result['completed'] > 0) {
        return round(($result['refunded'] / $result['completed']) * 100, 2);
    }
    return 0;
}

function getSuccessRate() {
    $db = getDB();
    $stmt = $db->query("SELECT 
        (SELECT COUNT(*) FROM revenue_transactions WHERE status = 'completed') as success,
        (SELECT COUNT(*) FROM revenue_transactions) as total");
    $result = $stmt->fetch();
    if ($result['total'] > 0) {
        return round(($result['success'] / $result['total']) * 100, 2);
    }
    return 0;
}

function getPeakHours() {
    $db = getDB();
    $stmt = $db->query("SELECT HOUR(created_at) as hour, COUNT(*) as count 
                       FROM revenue_transactions 
                       WHERE status = 'completed' 
                       GROUP BY HOUR(created_at) 
                       ORDER BY count DESC LIMIT 1");
    $result = $stmt->fetch();
    if ($result) {
        $hour = intval($result['hour']);
        return sprintf('%02d:00 - %02d:00', $hour, ($hour + 1) % 24);
    }
    return 'N/A';
}

function getTopCountry() {
    $db = getDB();
    $stmt = $db->query("SELECT u.country, COUNT(*) as count 
                       FROM revenue_transactions t 
                       JOIN app_users u ON t.user_id = u.user_id 
                       WHERE t.status = 'completed' AND u.country IS NOT NULL 
                       GROUP BY u.country ORDER BY count DESC LIMIT 1");
    $result = $stmt->fetch();
    return $result['country'] ?? 'N/A';
}
?>