<?php
// notifications.php - Push Notifications Management
session_start();
require_once 'includes/config.php';
require_once 'includes/functions.php';

checkAdminLogin();

$message = '';
$error = '';

// Handle send notification
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['send_notification'])) {
    $title = sanitize_input($_POST['title'] ?? '');
    $message_text = sanitize_input($_POST['message'] ?? '');
    $target = sanitize_input($_POST['target'] ?? 'all');
    
    if (empty($title) || empty($message_text)) {
        $error = 'Please fill all required fields';
    } else {
        if (sendPushNotification($title, $message_text, $target)) {
            $message = 'Notification sent successfully!';
        } else {
            $error = 'Failed to send notification';
        }
    }
}

// Handle delete notification
if (isset($_GET['delete'])) {
    $id = intval($_GET['delete']);
    $db = getDB();
    $stmt = $db->prepare("DELETE FROM push_notifications WHERE id = ?");
    if ($stmt->execute([$id])) {
        $message = 'Notification deleted successfully!';
    } else {
        $error = 'Failed to delete notification';
    }
}

// Get notifications
$notifications = getNotifications(50);

// Get user count for stats
$totalUsers = getTotalUsers();
$vipUsers = getVipUsers();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Push Notifications - Pink Video Chat Admin</title>
    <?php include 'includes/head.php'; ?>
    <style>
        .notification-card {
            border-left: 4px solid #FF69B4;
            margin-bottom: 15px;
        }
        .notification-card.sent {
            border-left-color: #28a745;
        }
        .notification-card.failed {
            border-left-color: #dc3545;
        }
        .notification-card.pending {
            border-left-color: #ffc107;
        }
        .target-badge {
            font-size: 0.8rem;
            padding: 2px 8px;
        }
    </style>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container mt-4">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="col-md-9">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h3><i class="fas fa-bell me-2"></i> Push Notifications</h3>
                    <button class="btn btn-pink" data-bs-toggle="modal" data-bs-target="#sendNotificationModal">
                        <i class="fas fa-plus me-1"></i> Send New Notification
                    </button>
                </div>
                
                <?php if ($message): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <?php echo $message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <?php echo $error; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>
                
                <div class="row mb-4">
                    <div class="col-md-4">
                        <div class="card stat-card bg-primary text-white">
                            <div class="card-body text-center">
                                <i class="fas fa-users fa-2x mb-2"></i>
                                <h4><?php echo number_format($totalUsers); ?></h4>
                                <p>Total Users</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card stat-card bg-success text-white">
                            <div class="card-body text-center">
                                <i class="fas fa-crown fa-2x mb-2"></i>
                                <h4><?php echo number_format($vipUsers); ?></h4>
                                <p>VIP Users</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card stat-card bg-info text-white">
                            <div class="card-body text-center">
                                <i class="fas fa-bell fa-2x mb-2"></i>
                                <h4><?php echo count($notifications); ?></h4>
                                <p>Notifications Sent</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Send Notification Form -->
                <div class="card mb-4">
                    <div class="card-header bg-pink text-white">
                        <h5 class="mb-0"><i class="fas fa-paper-plane me-2"></i> Send New Notification</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" id="notificationForm">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Notification Title *</label>
                                    <input type="text" class="form-control" name="title" 
                                           placeholder="Enter notification title" required>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Target Audience *</label>
                                    <select class="form-control" name="target" required>
                                        <option value="all">All Users</option>
                                        <option value="vip">VIP Users Only</option>
                                        <option value="non_vip">Non-VIP Users Only</option>
                                        <option value="new">New Users (Last 7 days)</option>
                                        <option value="inactive">Inactive Users (7+ days)</option>
                                    </select>
                                </div>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Notification Message *</label>
                                <textarea class="form-control" name="message" rows="4" 
                                          placeholder="Enter your message here..." required></textarea>
                                <small class="text-muted">Max 200 characters recommended</small>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Schedule (Optional)</label>
                                <input type="datetime-local" class="form-control" name="schedule_time">
                                <small class="text-muted">Leave empty to send immediately</small>
                            </div>
                            <button type="submit" name="send_notification" class="btn btn-pink">
                                <i class="fas fa-paper-plane me-2"></i> Send Notification
                            </button>
                            <button type="button" class="btn btn-outline-secondary" onclick="previewNotification()">
                                <i class="fas fa-eye me-2"></i> Preview
                            </button>
                        </form>
                    </div>
                </div>
                
                <!-- Notification History -->
                <div class="card">
                    <div class="card-header bg-pink text-white">
                        <h5 class="mb-0"><i class="fas fa-history me-2"></i> Notification History</h5>
                    </div>
                    <div class="card-body">
                        <?php if (empty($notifications)): ?>
                        <div class="text-center py-5">
                            <i class="fas fa-bell-slash fa-3x text-muted mb-3"></i>
                            <p class="text-muted">No notifications sent yet</p>
                        </div>
                        <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Title</th>
                                        <th>Target</th>
                                        <th>Status</th>
                                        <th>Sent At</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($notifications as $notification): ?>
                                    <tr>
                                        <td><?php echo $notification['id']; ?></td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($notification['title']); ?></strong><br>
                                            <small class="text-muted"><?php echo substr($notification['message'], 0, 50); ?>...</small>
                                        </td>
                                        <td>
                                            <span class="badge bg-info target-badge">
                                                <?php echo ucfirst($notification['target']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php 
                                                echo $notification['status'] == 'sent' ? 'success' : 
                                                     ($notification['status'] == 'failed' ? 'danger' : 'warning');
                                            ?>">
                                                <?php echo ucfirst($notification['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php echo formatDate($notification['sent_at'] ?? $notification['created_at']); ?>
                                        </td>
                                        <td>
                                            <button class="btn btn-sm btn-outline-info me-1" 
                                                    onclick="viewNotification(<?php echo $notification['id']; ?>)"
                                                    title="View Details">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                            <button class="btn btn-sm btn-outline-danger" 
                                                    onclick="deleteNotification(<?php echo $notification['id']; ?>)"
                                                    title="Delete">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Preview Modal -->
    <div class="modal fade" id="previewModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-pink text-white">
                    <h5 class="modal-title">Notification Preview</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="card">
                        <div class="card-body">
                            <h5 id="previewTitle" class="card-title"></h5>
                            <p id="previewMessage" class="card-text"></p>
                            <small class="text-muted" id="previewTarget"></small>
                        </div>
                    </div>
                    <div class="mt-3">
                        <p class="text-muted">This is how the notification will appear on users' devices.</p>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/scripts.php'; ?>
    <script>
    function previewNotification() {
        var title = document.querySelector('input[name="title"]').value;
        var message = document.querySelector('textarea[name="message"]').value;
        var target = document.querySelector('select[name="target"]').value;
        
        if (!title || !message) {
            alert('Please fill title and message first');
            return;
        }
        
        document.getElementById('previewTitle').textContent = title;
        document.getElementById('previewMessage').textContent = message;
        document.getElementById('previewTarget').textContent = 'Target: ' + target.charAt(0).toUpperCase() + target.slice(1) + ' Users';
        
        var previewModal = new bootstrap.Modal(document.getElementById('previewModal'));
        previewModal.show();
    }
    
    function viewNotification(id) {
        // In real app, load notification details via AJAX
        alert('Viewing notification ID: ' + id + '\nThis would show full details in a real application.');
    }
    
    function deleteNotification(id) {
        if (confirm('Are you sure you want to delete this notification?')) {
            window.location.href = 'notifications.php?delete=' + id;
        }
    }
    
    // Auto-count characters
    document.querySelector('textarea[name="message"]').addEventListener('input', function() {
        var charCount = this.value.length;
        var counter = this.parentElement.querySelector('.text-muted');
        if (charCount > 200) {
            counter.innerHTML = '<span class="text-danger">' + charCount + '/200 characters (too long)</span>';
        } else {
            counter.textContent = charCount + '/200 characters';
        }
    });
    </script>
</body>
</html>