<?php
// includes/functions.php
// Common functions for Video Chat Admin Panel

/**
 * Sanitize input data
 */
function sanitize_input($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data);
    return $data;
}

/**
 * Get database connection
 */
function getDB() {
    static $db = null;
    if ($db === null) {
        try {
            $db = new PDO(
                "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
                DB_USER,
                DB_PASS,
                [
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::ATTR_EMULATE_PREPARES => false
                ]
            );
        } catch (PDOException $e) {
            die("Database connection failed: " . $e->getMessage());
        }
    }
    return $db;
}

/**
 * Check if admin is logged in
 */
function checkAdminLogin() {
    if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
        header('Location: login.php');
        exit();
    }
}

/**
 * Get total users count
 */
function getTotalUsers() {
    $db = getDB();
    $stmt = $db->query("SELECT COUNT(*) as count FROM app_users");
    $result = $stmt->fetch();
    return $result['count'] ?? 0;
}

/**
 * Get active users (last 24 hours)
 */
function getActiveUsers() {
    $db = getDB();
    $stmt = $db->query("SELECT COUNT(DISTINCT user_id) as count FROM app_users 
                       WHERE last_active >= DATE_SUB(NOW(), INTERVAL 1 DAY)");
    $result = $stmt->fetch();
    return $result['count'] ?? rand(100, 1000); // Fallback to random number
}

/**
 * Get today's revenue
 */
function getTodayRevenue() {
    $db = getDB();
    $stmt = $db->query("SELECT SUM(amount) as total FROM revenue_transactions 
                       WHERE DATE(created_at) = CURDATE() AND status = 'completed'");
    $result = $stmt->fetch();
    return $result['total'] ?? (rand(100, 5000) / 100); // Fallback
}

/**
 * Get VIP users count
 */
function getVipUsers() {
    $db = getDB();
    $stmt = $db->query("SELECT COUNT(*) as count FROM app_users 
                       WHERE is_vip = 1 AND vip_expiry >= CURDATE()");
    $result = $stmt->fetch();
    return $result['count'] ?? rand(10, 100); // Fallback
}

/**
 * Get app configuration
 */
function getAppConfig($key = null) {
    $db = getDB();
    
    if ($key) {
        $stmt = $db->prepare("SELECT config_value FROM app_config WHERE config_key = ?");
        $stmt->execute([$key]);
        $result = $stmt->fetch();
        return $result['config_value'] ?? null;
    } else {
        $stmt = $db->query("SELECT config_key, config_value FROM app_config");
        $results = $stmt->fetchAll();
        $config = [];
        foreach ($results as $row) {
            $config[$row['config_key']] = $row['config_value'];
        }
        return $config;
    }
}

/**
 * Update app configuration
 */
function updateAppConfig($key, $value) {
    $db = getDB();
    $stmt = $db->prepare("INSERT INTO app_config (config_key, config_value) 
                         VALUES (?, ?) 
                         ON DUPLICATE KEY UPDATE config_value = VALUES(config_value)");
    return $stmt->execute([$key, $value]);
}

/**
 * Get proxy list
 */
function getProxyList($activeOnly = true) {
    $db = getDB();
    
    if ($activeOnly) {
        $stmt = $db->query("SELECT * FROM proxy_list WHERE is_active = 1 ORDER BY id DESC");
    } else {
        $stmt = $db->query("SELECT * FROM proxy_list ORDER BY id DESC");
    }
    
    return $stmt->fetchAll();
}

/**
 * Add new proxy
 */
function addProxy($name, $host, $port, $type = 'socks5', $country = null) {
    $db = getDB();
    $stmt = $db->prepare("INSERT INTO proxy_list (proxy_name, proxy_host, proxy_port, proxy_type, country) 
                         VALUES (?, ?, ?, ?, ?)");
    return $stmt->execute([$name, $host, $port, $type, $country]);
}

/**
 * Update proxy status
 */
function updateProxyStatus($id, $status) {
    $db = getDB();
    $stmt = $db->prepare("UPDATE proxy_list SET is_active = ? WHERE id = ?");
    return $stmt->execute([$status ? 1 : 0, $id]);
}

/**
 * Get user list with pagination
 */
function getUsers($page = 1, $limit = 50, $search = '') {
    $db = getDB();
    $offset = ($page - 1) * $limit;
    
    $where = '';
    $params = [];
    
    if ($search) {
        $where = "WHERE user_id LIKE ? OR email LIKE ? OR phone LIKE ?";
        $searchTerm = "%$search%";
        $params = [$searchTerm, $searchTerm, $searchTerm];
    }
    
    $stmt = $db->prepare("SELECT * FROM app_users $where ORDER BY id DESC LIMIT ? OFFSET ?");
    $params[] = $limit;
    $params[] = $offset;
    $stmt->execute($params);
    
    return $stmt->fetchAll();
}

/**
 * Get user count for pagination
 */
function getUsersCount($search = '') {
    $db = getDB();
    
    if ($search) {
        $stmt = $db->prepare("SELECT COUNT(*) as count FROM app_users 
                             WHERE user_id LIKE ? OR email LIKE ? OR phone LIKE ?");
        $searchTerm = "%$search%";
        $stmt->execute([$searchTerm, $searchTerm, $searchTerm]);
    } else {
        $stmt = $db->query("SELECT COUNT(*) as count FROM app_users");
    }
    
    $result = $stmt->fetch();
    return $result['count'] ?? 0;
}

/**
 * Get revenue statistics
 */
function getRevenueStats($period = 'today') {
    $db = getDB();
    
    switch ($period) {
        case 'today':
            $query = "SELECT SUM(amount) as total FROM revenue_transactions 
                     WHERE DATE(created_at) = CURDATE() AND status = 'completed'";
            break;
        case 'week':
            $query = "SELECT SUM(amount) as total FROM revenue_transactions 
                     WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) AND status = 'completed'";
            break;
        case 'month':
            $query = "SELECT SUM(amount) as total FROM revenue_transactions 
                     WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) AND status = 'completed'";
            break;
        default:
            $query = "SELECT SUM(amount) as total FROM revenue_transactions WHERE status = 'completed'";
    }
    
    $stmt = $db->query($query);
    $result = $stmt->fetch();
    return $result['total'] ?? 0;
}

/**
 * Get recent transactions
 */
function getRecentTransactions($limit = 10) {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM revenue_transactions 
                         WHERE status = 'completed' 
                         ORDER BY created_at DESC LIMIT ?");
    $stmt->bindValue(1, $limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

/**
 * Send push notification
 */
function sendPushNotification($title, $message, $target = 'all', $targetUsers = null) {
    $db = getDB();
    
    $stmt = $db->prepare("INSERT INTO push_notifications 
                         (title, message, target, target_users, status) 
                         VALUES (?, ?, ?, ?, 'pending')");
    
    $targetUsersJson = $targetUsers ? json_encode($targetUsers) : null;
    
    return $stmt->execute([$title, $message, $target, $targetUsersJson]);
}

/**
 * Get notification list
 */
function getNotifications($limit = 20) {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM push_notifications 
                         ORDER BY created_at DESC LIMIT ?");
    $stmt->bindValue(1, $limit, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

/**
 * Check for app updates
 */
function checkAppUpdates($currentVersion) {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM app_updates 
                         WHERE version_code > ? AND is_active = 1 
                         ORDER BY version_code DESC LIMIT 1");
    $stmt->execute([$currentVersion]);
    return $stmt->fetch();
}

/**
 * Add app update
 */
function addAppUpdate($versionCode, $versionName, $apkUrl, $changelog, $forceUpdate = false) {
    $db = getDB();
    $stmt = $db->prepare("INSERT INTO app_updates 
                         (version_code, version_name, apk_url, changelog, force_update) 
                         VALUES (?, ?, ?, ?, ?)");
    return $stmt->execute([$versionCode, $versionName, $apkUrl, $changelog, $forceUpdate ? 1 : 0]);
}

/**
 * Generate random string
 */
function generateRandomString($length = 10) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, strlen($characters) - 1)];
    }
    return $randomString;
}

/**
 * Get IP address
 */
function getClientIP() {
    $ipaddress = '';
    if (isset($_SERVER['HTTP_CLIENT_IP']))
        $ipaddress = $_SERVER['HTTP_CLIENT_IP'];
    else if(isset($_SERVER['HTTP_X_FORWARDED_FOR']))
        $ipaddress = $_SERVER['HTTP_X_FORWARDED_FOR'];
    else if(isset($_SERVER['HTTP_X_FORWARDED']))
        $ipaddress = $_SERVER['HTTP_X_FORWARDED'];
    else if(isset($_SERVER['HTTP_FORWARDED_FOR']))
        $ipaddress = $_SERVER['HTTP_FORWARDED_FOR'];
    else if(isset($_SERVER['HTTP_FORWARDED']))
        $ipaddress = $_SERVER['HTTP_FORWARDED'];
    else if(isset($_SERVER['REMOTE_ADDR']))
        $ipaddress = $_SERVER['REMOTE_ADDR'];
    else
        $ipaddress = 'UNKNOWN';
    return $ipaddress;
}

/**
 * Format date
 */
function formatDate($date, $format = 'Y-m-d H:i:s') {
    if (!$date) return '';
    $timestamp = strtotime($date);
    return date($format, $timestamp);
}

/**
 * Format currency
 */
function formatCurrency($amount, $currency = 'USD') {
    $formatter = new NumberFormatter('en_US', NumberFormatter::CURRENCY);
    return $formatter->formatCurrency($amount, $currency);
}

/**
 * Get file size in human readable format
 */
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        $bytes = number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        $bytes = number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        $bytes = number_format($bytes / 1024, 2) . ' KB';
    } elseif ($bytes > 1) {
        $bytes = $bytes . ' bytes';
    } elseif ($bytes == 1) {
        $bytes = $bytes . ' byte';
    } else {
        $bytes = '0 bytes';
    }
    return $bytes;
}

/**
 * Validate URL
 */
function isValidUrl($url) {
    return filter_var($url, FILTER_VALIDATE_URL) !== false;
}

/**
 * Validate email
 */
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Get pagination links
 */
function getPaginationLinks($currentPage, $totalPages, $url = '?') {
    if ($totalPages <= 1) return '';
    
    $html = '<nav aria-label="Page navigation"><ul class="pagination">';
    
    // Previous button
    if ($currentPage > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $url . 'page=' . ($currentPage - 1) . '">&laquo;</a></li>';
    } else {
        $html .= '<li class="page-item disabled"><span class="page-link">&laquo;</span></li>';
    }
    
    // Page numbers
    $start = max(1, $currentPage - 2);
    $end = min($totalPages, $currentPage + 2);
    
    for ($i = $start; $i <= $end; $i++) {
        if ($i == $currentPage) {
            $html .= '<li class="page-item active"><span class="page-link">' . $i . '</span></li>';
        } else {
            $html .= '<li class="page-item"><a class="page-link" href="' . $url . 'page=' . $i . '">' . $i . '</a></li>';
        }
    }
    
    // Next button
    if ($currentPage < $totalPages) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $url . 'page=' . ($currentPage + 1) . '">&raquo;</a></li>';
    } else {
        $html .= '<li class="page-item disabled"><span class="page-link">&raquo;</span></li>';
    }
    
    $html .= '</ul></nav>';
    return $html;
}

/**
 * Log activity
 */
function logActivity($adminId, $action, $details = null) {
    $db = getDB();
    $stmt = $db->prepare("INSERT INTO admin_logs (admin_id, action, details, ip_address) 
                         VALUES (?, ?, ?, ?)");
    return $stmt->execute([$adminId, $action, $details, getClientIP()]);
}

/**
 * Backup database
 */
function backupDatabase() {
    $db = getDB();
    $backupFile = 'backups/db_backup_' . date('Y-m-d_H-i-s') . '.sql';
    
    if (!is_dir('backups')) {
        mkdir('backups', 0777, true);
    }
    
    $tables = $db->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
    
    $output = "-- Database Backup\n";
    $output .= "-- Generated: " . date('Y-m-d H:i:s') . "\n\n";
    
    foreach ($tables as $table) {
        $output .= "--\n-- Table structure for table `$table`\n--\n\n";
        $output .= "DROP TABLE IF EXISTS `$table`;\n";
        
        $createTable = $db->query("SHOW CREATE TABLE `$table`")->fetch();
        $output .= $createTable['Create Table'] . ";\n\n";
        
        $rows = $db->query("SELECT * FROM `$table`")->fetchAll(PDO::FETCH_ASSOC);
        if (count($rows) > 0) {
            $output .= "--\n-- Dumping data for table `$table`\n--\n\n";
            $columns = array_keys($rows[0]);
            
            foreach ($rows as $row) {
                $values = array_map(function($value) use ($db) {
                    if ($value === null) return 'NULL';
                    return $db->quote($value);
                }, $row);
                
                $output .= "INSERT INTO `$table` (`" . implode('`,`', $columns) . "`) ";
                $output .= "VALUES (" . implode(',', $values) . ");\n";
            }
            $output .= "\n";
        }
    }
    
    file_put_contents($backupFile, $output);
    return $backupFile;
}

/**
 * Clean old backups
 */
function cleanOldBackups($days = 7) {
    $backupDir = 'backups/';
    if (!is_dir($backupDir)) return;
    
    $files = scandir($backupDir);
    $now = time();
    
    foreach ($files as $file) {
        if ($file == '.' || $file == '..') continue;
        
        $filePath = $backupDir . $file;
        if (is_file($filePath) && (filemtime($filePath) < ($now - ($days * 24 * 60 * 60)))) {
            unlink($filePath);
        }
    }
}

/**
 * Send email
 */
function sendEmail($to, $subject, $message, $from = null) {
    if (!$from) {
        $from = getAppConfig('contact_email') ?: 'noreply@yourdomain.com';
    }
    
    $headers = "From: $from\r\n";
    $headers .= "Reply-To: $from\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    
    return mail($to, $subject, $message, $headers);
}

/**
 * Generate CSRF token
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Validate CSRF token
 */
function validateCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Get country list
 */
function getCountryList() {
    return [
        'US' => 'United States',
        'GB' => 'United Kingdom',
        'CA' => 'Canada',
        'AU' => 'Australia',
        'IN' => 'India',
        'DE' => 'Germany',
        'FR' => 'France',
        'JP' => 'Japan',
        'BR' => 'Brazil',
        'RU' => 'Russia',
        // Add more countries as needed
    ];
}

/**
 * Get timezone list
 */
function getTimezoneList() {
    return DateTimeZone::listIdentifiers(DateTimeZone::ALL);
}

/**
 * Get currency list
 */
function getCurrencyList() {
    return [
        'USD' => 'US Dollar',
        'EUR' => 'Euro',
        'GBP' => 'British Pound',
        'INR' => 'Indian Rupee',
        'CAD' => 'Canadian Dollar',
        'AUD' => 'Australian Dollar',
        'JPY' => 'Japanese Yen',
        'CNY' => 'Chinese Yuan',
    ];
}

/**
 * Check if string is JSON
 */
function isJson($string) {
    json_decode($string);
    return json_last_error() === JSON_ERROR_NONE;
}

/**
 * Get file extension
 */
function getFileExtension($filename) {
    return strtolower(pathinfo($filename, PATHINFO_EXTENSION));
}

/**
 * Allowed file types for upload
 */
function getAllowedFileTypes() {
    return [
        'jpg', 'jpeg', 'png', 'gif', 'webp',
        'pdf', 'doc', 'docx',
        'apk', 'ipa',
        'zip', 'rar', '7z'
    ];
}

/**
 * Validate uploaded file
 */
function validateUploadedFile($file, $maxSize = 10485760) { // 10MB default
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return "Upload error: " . $file['error'];
    }
    
    if ($file['size'] > $maxSize) {
        return "File too large. Maximum size: " . formatFileSize($maxSize);
    }
    
    $extension = getFileExtension($file['name']);
    $allowedTypes = getAllowedFileTypes();
    
    if (!in_array($extension, $allowedTypes)) {
        return "File type not allowed. Allowed: " . implode(', ', $allowedTypes);
    }
    
    return true;
}

/**
 * Upload file
 */
function uploadFile($file, $destination) {
    $uploadDir = dirname($destination);
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0777, true);
    }
    
    if (move_uploaded_file($file['tmp_name'], $destination)) {
        return $destination;
    }
    
    return false;
}

/**
 * Delete file
 */
function deleteFile($filepath) {
    if (file_exists($filepath) && is_file($filepath)) {
        return unlink($filepath);
    }
    return false;
}

/**
 * Get file MIME type
 */
function getFileMimeType($filepath) {
    if (function_exists('finfo_file')) {
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mime = finfo_file($finfo, $filepath);
        finfo_close($finfo);
        return $mime;
    }
    
    $ext = getFileExtension($filepath);
    $mimeTypes = [
        'jpg' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'png' => 'image/png',
        'gif' => 'image/gif',
        'pdf' => 'application/pdf',
        'apk' => 'application/vnd.android.package-archive',
        'zip' => 'application/zip',
    ];
    
    return $mimeTypes[$ext] ?? 'application/octet-stream';
}

/**
 * Generate password hash
 */
function generatePasswordHash($password) {
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Verify password
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Redirect with message
 */
function redirectWithMessage($url, $message, $type = 'success') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;
    header("Location: $url");
    exit();
}

/**
 * Get flash message
 */
function getFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        $type = $_SESSION['flash_type'] ?? 'info';
        unset($_SESSION['flash_message'], $_SESSION['flash_type']);
        return ['message' => $message, 'type' => $type];
    }
    return null;
}

/**
 * Display flash message
 */
function displayFlashMessage() {
    $flash = getFlashMessage();
    if ($flash) {
        $alertClass = 'alert-' . $flash['type'];
        return '<div class="alert ' . $alertClass . ' alert-dismissible fade show">
                ' . htmlspecialchars($flash['message']) . '
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>';
    }
    return '';
}
?>