<?php
// project_generator.php
// Upload this file to CPanel and run it to generate complete Video Chat project

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

$baseDir = __DIR__ . '/videochat_project';

function createFile($path, $content) {
    $dir = dirname($path);
    if (!is_dir($dir)) {
        mkdir($dir, 0777, true);
    }
    file_put_contents($path, $content);
    return $path;
}

function createAndroidProject() {
    global $baseDir;
    
    $androidDir = $baseDir . '/android_app';
    
    // Create MainActivity.java
    $mainActivity = <<<'JAVA'
package com.pubvance.videocall;

import android.annotation.SuppressLint;
import android.content.Intent;
import android.content.SharedPreferences;
import android.graphics.Color;
import android.os.Bundle;
import android.view.View;
import android.webkit.*;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.TextView;
import androidx.appcompat.app.AppCompatActivity;
import okhttp3.*;
import org.json.JSONObject;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

public class MainActivity extends AppCompatActivity {
    
    private WebView webView;
    private OkHttpClient okHttpClient;
    private SharedPreferences prefs;
    private FrameLayout webViewContainer;
    private Button startBtn, adminBtn;
    
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        
        prefs = getSharedPreferences("VideoChatPrefs", MODE_PRIVATE);
        
        // Initialize views
        webViewContainer = findViewById(R.id.webViewContainer);
        startBtn = findViewById(R.id.startBtn);
        adminBtn = findViewById(R.id.adminBtn);
        
        // Set pink theme
        startBtn.setBackgroundColor(Color.parseColor("#FF69B4"));
        adminBtn.setBackgroundColor(Color.parseColor("#FFC0CB"));
        
        // Load saved URL or use default
        String defaultUrl = prefs.getString("webview_url", "https://google.com");
        
        // Setup proxy
        setupProxy();
        
        // Setup buttons
        startBtn.setOnClickListener(v -> {
            startBtn.setText("Connecting...");
            startBtn.setEnabled(false);
            initializeWebView(defaultUrl);
            webViewContainer.setVisibility(View.VISIBLE);
        });
        
        adminBtn.setOnLongClickListener(v -> {
            Intent intent = new Intent(MainActivity.this, AdminActivity.class);
            startActivity(intent);
            return true;
        });
    }
    
    private void setupProxy() {
        String proxyHost = prefs.getString("proxy_host", "your-proxy.com");
        int proxyPort = prefs.getInt("proxy_port", 1080);
        
        Proxy proxy = new Proxy(Proxy.Type.SOCKS, 
            new InetSocketAddress(proxyHost, proxyPort));
        
        okHttpClient = new OkHttpClient.Builder()
            .proxy(proxy)
            .connectTimeout(30, TimeUnit.SECONDS)
            .readTimeout(30, TimeUnit.SECONDS)
            .addInterceptor(chain -> {
                Request original = chain.request();
                Request request = original.newBuilder()
                    .header("User-Agent", WebSettings.getDefaultUserAgent(this))
                    .build();
                return chain.proceed(request);
            })
            .build();
    }
    
    @SuppressLint("SetJavaScriptEnabled")
    private void initializeWebView(String url) {
        webView = new WebView(this);
        webViewContainer.addView(webView);
        
        WebSettings settings = webView.getSettings();
        settings.setJavaScriptEnabled(true);
        settings.setDomStorageEnabled(true);
        settings.setMediaPlaybackRequiresUserGesture(false);
        
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
            settings.setMixedContentMode(WebSettings.MIXED_CONTENT_ALWAYS_ALLOW);
        }
        
        webView.setWebViewClient(new WebViewClient() {
            @Override
            public boolean shouldOverrideUrlLoading(WebView view, String url) {
                view.loadUrl(url);
                return true;
            }
        });
        
        webView.loadUrl(url);
    }
    
    @Override
    public void onBackPressed() {
        if (webView != null && webView.canGoBack()) {
            webView.goBack();
        } else if (webViewContainer.getVisibility() == View.VISIBLE) {
            webViewContainer.setVisibility(View.GONE);
            startBtn.setVisibility(View.VISIBLE);
            startBtn.setText("Start Video Chat");
            startBtn.setEnabled(true);
        } else {
            super.onBackPressed();
        }
    }
}
JAVA;
    
    createFile($androidDir . '/app/src/main/java/com/pubvance/videocall/MainActivity.java', $mainActivity);
    
    // Create AdminActivity.java
    $adminActivity = <<<'JAVA'
package com.pubvance.videocall;

import android.content.SharedPreferences;
import android.os.Bundle;
import android.view.View;
import android.widget.*;
import androidx.appcompat.app.AppCompatActivity;

public class AdminActivity extends AppCompatActivity {
    
    private EditText urlEditText, proxyHostEditText, proxyPortEditText;
    private Button saveBtn;
    private SharedPreferences prefs;
    
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_admin);
        
        prefs = getSharedPreferences("VideoChatPrefs", MODE_PRIVATE);
        
        urlEditText = findViewById(R.id.urlEditText);
        proxyHostEditText = findViewById(R.id.proxyHostEditText);
        proxyPortEditText = findViewById(R.id.proxyPortEditText);
        saveBtn = findViewById(R.id.saveBtn);
        
        // Load saved settings
        urlEditText.setText(prefs.getString("webview_url", "https://google.com"));
        proxyHostEditText.setText(prefs.getString("proxy_host", "your-proxy.com"));
        proxyPortEditText.setText(String.valueOf(prefs.getInt("proxy_port", 1080)));
        
        saveBtn.setOnClickListener(v -> saveSettings());
    }
    
    private void saveSettings() {
        String url = urlEditText.getText().toString();
        String proxyHost = proxyHostEditText.getText().toString();
        String proxyPortStr = proxyPortEditText.getText().toString();
        
        if (url.isEmpty() || proxyHost.isEmpty() || proxyPortStr.isEmpty()) {
            Toast.makeText(this, "Please fill all fields", Toast.LENGTH_SHORT).show();
            return;
        }
        
        try {
            int proxyPort = Integer.parseInt(proxyPortStr);
            
            prefs.edit()
                .putString("webview_url", url)
                .putString("proxy_host", proxyHost)
                .putInt("proxy_port", proxyPort)
                .apply();
            
            Toast.makeText(this, "Settings saved!", Toast.LENGTH_SHORT).show();
            finish();
            
        } catch (NumberFormatException e) {
            Toast.makeText(this, "Invalid port number", Toast.LENGTH_SHORT).show();
        }
    }
}
JAVA;
    
    createFile($androidDir . '/app/src/main/java/com/pubvance/videocall/AdminActivity.java', $adminActivity);
    
    // Create AndroidManifest.xml
    $manifest = <<<'XML'
<?xml version="1.0" encoding="utf-8"?>
<manifest xmlns:android="http://schemas.android.com/apk/res/android"
    package="com.pubvance.videocall">

    <uses-permission android:name="android.permission.INTERNET" />
    <uses-permission android:name="android.permission.ACCESS_NETWORK_STATE" />
    <uses-permission android:name="android.permission.CAMERA" />
    <uses-permission android:name="android.permission.RECORD_AUDIO" />

    <application
        android:allowBackup="true"
        android:icon="@mipmap/ic_launcher"
        android:label="Pink Video Chat"
        android:theme="@style/Theme.AppCompat.Light.NoActionBar"
        android:usesCleartextTraffic="true"
        android:hardwareAccelerated="true">

        <activity
            android:name=".MainActivity"
            android:exported="true">
            <intent-filter>
                <action android:name="android.intent.action.MAIN" />
                <category android:name="android.intent.category.LAUNCHER" />
            </intent-filter>
        </activity>

        <activity
            android:name=".AdminActivity"
            android:exported="false" />

    </application>

</manifest>
XML;
    
    createFile($androidDir . '/app/src/main/AndroidManifest.xml', $manifest);
    
    // Create activity_main.xml
    $activityMain = <<<'XML'
<?xml version="1.0" encoding="utf-8"?>
<RelativeLayout xmlns:android="http://schemas.android.com/apk/res/android"
    android:layout_width="match_parent"
    android:layout_height="match_parent"
    android:background="#FFC0CB"
    android:padding="16dp">

    <LinearLayout
        android:layout_width="match_parent"
        android:layout_height="wrap_content"
        android:orientation="vertical"
        android:gravity="center"
        android:layout_centerInParent="true"
        android:padding="20dp">

        <TextView
            android:layout_width="wrap_content"
            android:layout_height="wrap_content"
            android:text="Pink Video Chat"
            android:textSize="28sp"
            android:textColor="#FF1493"
            android:textStyle="bold"
            android:layout_marginBottom="30dp" />

        <Button
            android:id="@+id/startBtn"
            android:layout_width="match_parent"
            android:layout_height="60dp"
            android:text="Start Video Chat"
            android:textSize="18sp"
            android:layout_marginBottom="20dp" />

        <Button
            android:id="@+id/adminBtn"
            android:layout_width="match_parent"
            android:layout_height="50dp"
            android:text="Admin Settings"
            android:textSize="14sp" />

    </LinearLayout>

    <FrameLayout
        android:id="@+id/webViewContainer"
        android:layout_width="match_parent"
        android:layout_height="match_parent"
        android:visibility="gone" />

</RelativeLayout>
XML;
    
    createFile($androidDir . '/app/src/main/res/layout/activity_main.xml', $activityMain);
    
    // Create activity_admin.xml
    $activityAdmin = <<<'XML'
<?xml version="1.0" encoding="utf-8"?>
<ScrollView xmlns:android="http://schemas.android.com/apk/res/android"
    android:layout_width="match_parent"
    android:layout_height="match_parent"
    android:padding="20dp"
    android:background="#FFC0CB">

    <LinearLayout
        android:layout_width="match_parent"
        android:layout_height="wrap_content"
        android:orientation="vertical">

        <TextView
            android:layout_width="match_parent"
            android:layout_height="wrap_content"
            android:text="Admin Settings"
            android:textSize="24sp"
            android:textColor="#FF1493"
            android:textStyle="bold"
            android:gravity="center"
            android:layout_marginBottom="30dp" />

        <TextView
            android:layout_width="wrap_content"
            android:layout_height="wrap_content"
            android:text="WebView URL"
            android:textSize="16sp"
            android:layout_marginBottom="5dp" />

        <EditText
            android:id="@+id/urlEditText"
            android:layout_width="match_parent"
            android:layout_height="50dp"
            android:hint="https://example.com"
            android:inputType="textUri"
            android:padding="10dp"
            android:background="#FFFFFF"
            android:layout_marginBottom="20dp" />

        <TextView
            android:layout_width="wrap_content"
            android:layout_height="wrap_content"
            android:text="SOCKS5 Proxy Host"
            android:textSize="16sp"
            android:layout_marginBottom="5dp" />

        <EditText
            android:id="@+id/proxyHostEditText"
            android:layout_width="match_parent"
            android:layout_height="50dp"
            android:hint="proxy.example.com"
            android:padding="10dp"
            android:background="#FFFFFF"
            android:layout_marginBottom="20dp" />

        <TextView
            android:layout_width="wrap_content"
            android:layout_height="wrap_content"
            android:text="Proxy Port"
            android:textSize="16sp"
            android:layout_marginBottom="5dp" />

        <EditText
            android:id="@+id/proxyPortEditText"
            android:layout_width="match_parent"
            android:layout_height="50dp"
            android:hint="1080"
            android:inputType="number"
            android:padding="10dp"
            android:background="#FFFFFF"
            android:layout_marginBottom="30dp" />

        <Button
            android:id="@+id/saveBtn"
            android:layout_width="match_parent"
            android:layout_height="50dp"
            android:text="Save Settings"
            android:background="#FF1493"
            android:textColor="#FFFFFF" />

    </LinearLayout>

</ScrollView>
XML;
    
    createFile($androidDir . '/app/src/main/res/layout/activity_admin.xml', $activityAdmin);
    
    // Create build.gradle
    $buildGradle = <<<'GRADLE'
plugins {
    id 'com.android.application'
}

android {
    compileSdk 34

    defaultConfig {
        applicationId "com.pubvance.videocall"
        minSdk 21
        targetSdk 34
        versionCode 1
        versionName "1.0"
    }

    buildTypes {
        release {
            minifyEnabled false
        }
    }
    
    compileOptions {
        sourceCompatibility JavaVersion.VERSION_1_8
        targetCompatibility JavaVersion.VERSION_1_8
    }
}

dependencies {
    implementation 'androidx.appcompat:appcompat:1.6.1'
    implementation 'com.google.android.material:material:1.9.0'
    implementation 'androidx.constraintlayout:constraintlayout:2.1.4'
    implementation 'com.squareup.okhttp3:okhttp:4.11.0'
}
GRADLE;
    
    createFile($androidDir . '/app/build.gradle', $buildGradle);
    
    // Create README.txt
    $readme = <<<'TEXT'
PINK VIDEO CHAT APP - ANDROID PROJECT

Instructions:
1. Open in Android Studio
2. Sync Gradle files
3. Build APK
4. Install on device

Features:
- WebView with SOCKS5 proxy
- Admin panel for URL changes
- Pink theme design
- Camera/microphone permissions
- Auto WebView loading

To modify:
- Change default URL in MainActivity.java
- Update proxy settings
- Modify colors in XML files
TEXT;
    
    createFile($androidDir . '/README.txt', $readme);
    
    return count(scandir_recursive($androidDir));
}

function createCpanelAdmin() {
    global $baseDir;
    
    $cpanelDir = $baseDir . '/cpanel_admin';
    
    // Create index.php (Main Dashboard)
    $index = <<<'PHP'
<?php
// index.php - Admin Dashboard
session_start();
require_once 'includes/config.php';
require_once 'includes/functions.php';

if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: login.php');
    exit();
}

$stats = [
    'total_users' => getTotalUsers(),
    'active_users' => getActiveUsers(),
    'today_revenue' => getTodayRevenue(),
    'vip_users' => getVipUsers()
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pink Video Chat - Admin Panel</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --pink: #FF69B4;
            --pink-dark: #FF1493;
            --pink-light: #FFC0CB;
        }
        body {
            background: linear-gradient(135deg, var(--pink-light), var(--pink));
            min-height: 100vh;
        }
        .navbar {
            background: var(--pink-dark);
        }
        .card {
            border-radius: 15px;
            border: none;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            transition: transform 0.3s;
        }
        .card:hover {
            transform: translateY(-5px);
        }
        .stat-card {
            color: white;
            text-align: center;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .bg-users { background: linear-gradient(45deg, #667eea, #764ba2); }
        .bg-revenue { background: linear-gradient(45deg, #f093fb, #f5576c); }
        .bg-active { background: linear-gradient(45deg, #4facfe, #00f2fe); }
        .bg-vip { background: linear-gradient(45deg, #43e97b, #38f9d7); }
        .btn-pink {
            background: var(--pink);
            color: white;
            border: none;
        }
        .btn-pink:hover {
            background: var(--pink-dark);
            color: white;
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="#">
                <i class="fas fa-video me-2"></i> Pink Video Chat Admin
            </a>
            <div class="d-flex align-items-center">
                <span class="text-white me-3">Welcome, <?php echo $_SESSION['admin_username']; ?></span>
                <a href="logout.php" class="btn btn-outline-light btn-sm">
                    <i class="fas fa-sign-out-alt"></i> Logout
                </a>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <div class="row">
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body p-0">
                        <div class="list-group list-group-flush">
                            <a href="index.php" class="list-group-item list-group-item-action active">
                                <i class="fas fa-tachometer-alt me-2"></i> Dashboard
                            </a>
                            <a href="users.php" class="list-group-item list-group-item-action">
                                <i class="fas fa-users me-2"></i> Users
                            </a>
                            <a href="revenue.php" class="list-group-item list-group-item-action">
                                <i class="fas fa-dollar-sign me-2"></i> Revenue
                            </a>
                            <a href="config.php" class="list-group-item list-group-item-action">
                                <i class="fas fa-cog me-2"></i> Settings
                            </a>
                            <a href="proxy.php" class="list-group-item list-group-item-action">
                                <i class="fas fa-server me-2"></i> Proxy
                            </a>
                            <a href="updates.php" class="list-group-item list-group-item-action">
                                <i class="fas fa-sync-alt me-2"></i> Updates
                            </a>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-9">
                <div class="row">
                    <div class="col-md-3">
                        <div class="stat-card bg-users">
                            <i class="fas fa-users fa-3x mb-3"></i>
                            <h4><?php echo $stats['total_users']; ?></h4>
                            <p>Total Users</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card bg-active">
                            <i class="fas fa-user-check fa-3x mb-3"></i>
                            <h4><?php echo $stats['active_users']; ?></h4>
                            <p>Active Users</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card bg-revenue">
                            <i class="fas fa-dollar-sign fa-3x mb-3"></i>
                            <h4>$<?php echo number_format($stats['today_revenue'], 2); ?></h4>
                            <p>Today's Revenue</p>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card bg-vip">
                            <i class="fas fa-crown fa-3x mb-3"></i>
                            <h4><?php echo $stats['vip_users']; ?></h4>
                            <p>VIP Users</p>
                        </div>
                    </div>
                </div>

                <div class="card mt-4">
                    <div class="card-header bg-pink text-white">
                        <h5 class="mb-0"><i class="fas fa-cog me-2"></i> Quick Actions</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <button class="btn btn-pink w-100" onclick="location.href='config.php'">
                                    <i class="fas fa-edit me-2"></i> Change WebView URL
                                </button>
                            </div>
                            <div class="col-md-4 mb-3">
                                <button class="btn btn-pink w-100" onclick="location.href='proxy.php'">
                                    <i class="fas fa-server me-2"></i> Manage Proxies
                                </button>
                            </div>
                            <div class="col-md-4 mb-3">
                                <button class="btn btn-pink w-100" onclick="location.href='updates.php'">
                                    <i class="fas fa-upload me-2"></i> Upload APK
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
PHP;
    
    createFile($cpanelDir . '/index.php', $index);
    
    // Create login.php
    $login = <<<'PHP'
<?php
// login.php
session_start();
if (isset($_SESSION['admin_logged_in'])) {
    header('Location: index.php');
    exit();
}

$error = '';
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';
    
    // Default admin credentials (change in production)
    if ($username === 'admin' && $password === 'admin123') {
        $_SESSION['admin_logged_in'] = true;
        $_SESSION['admin_username'] = $username;
        header('Location: index.php');
        exit();
    } else {
        $error = 'Invalid username or password';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Pink Video Chat Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #FFC0CB, #FF69B4);
            height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .login-box {
            background: white;
            padding: 40px;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
            width: 100%;
            max-width: 400px;
        }
        .btn-login {
            background: #FF1493;
            color: white;
            width: 100%;
            padding: 12px;
        }
        .btn-login:hover {
            background: #FF69B4;
            color: white;
        }
    </style>
</head>
<body>
    <div class="login-box">
        <h3 class="text-center mb-4 text-pink">
            <i class="fas fa-video me-2"></i> Pink Video Chat
        </h3>
        <p class="text-center text-muted mb-4">Admin Panel Login</p>
        
        <?php if ($error): ?>
        <div class="alert alert-danger"><?php echo $error; ?></div>
        <?php endif; ?>
        
        <form method="POST">
            <div class="mb-3">
                <label class="form-label">Username</label>
                <input type="text" name="username" class="form-control" required>
            </div>
            <div class="mb-3">
                <label class="form-label">Password</label>
                <input type="password" name="password" class="form-control" required>
            </div>
            <button type="submit" class="btn btn-login">
                <i class="fas fa-sign-in-alt me-2"></i> Login
            </button>
        </form>
        
        <div class="mt-3 text-center">
            <small class="text-muted">Default: admin / admin123</small>
        </div>
    </div>
    
    <script src="https://kit.fontawesome.com/your-fontawesome-kit.js" crossorigin="anonymous"></script>
</body>
</html>
PHP;
    
    createFile($cpanelDir . '/login.php', $login);
    
    // Create logout.php
    $logout = <<<'PHP'
<?php
// logout.php
session_start();
session_destroy();
header('Location: login.php');
exit();
PHP;
    
    createFile($cpanelDir . '/logout.php', $logout);
    
    // Create config.php
    $config = <<<'PHP'
<?php
// config.php - App Configuration
session_start();
require_once 'includes/config.php';
require_once 'includes/functions.php';

if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: login.php');
    exit();
}

$message = '';
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $webview_url = $_POST['webview_url'] ?? '';
    $app_name = $_POST['app_name'] ?? '';
    $contact_email = $_POST['contact_email'] ?? '';
    
    // Save to database or config file
    $configData = [
        'webview_url' => $webview_url,
        'app_name' => $app_name,
        'contact_email' => $contact_email,
        'updated_at' => date('Y-m-d H:i:s')
    ];
    
    // In real app, save to database
    file_put_contents('config/app_config.json', json_encode($configData, JSON_PRETTY_PRINT));
    
    $message = '<div class="alert alert-success">Settings saved successfully!</div>';
}

// Load current settings
$currentConfig = file_exists('config/app_config.json') 
    ? json_decode(file_get_contents('config/app_config.json'), true)
    : [
        'webview_url' => 'https://your-wordpress-site.com',
        'app_name' => 'Pink Video Chat',
        'contact_email' => 'support@yourdomain.com'
    ];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>App Settings - Admin Panel</title>
    <?php include 'includes/head.php'; ?>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container mt-4">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="col-md-9">
                <div class="card">
                    <div class="card-header bg-pink text-white">
                        <h5 class="mb-0"><i class="fas fa-cog me-2"></i> App Configuration</h5>
                    </div>
                    <div class="card-body">
                        <?php echo $message; ?>
                        
                        <form method="POST">
                            <div class="mb-3">
                                <label class="form-label">App Name</label>
                                <input type="text" name="app_name" class="form-control" 
                                       value="<?php echo htmlspecialchars($currentConfig['app_name']); ?>" required>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">WebView URL</label>
                                <input type="url" name="webview_url" class="form-control" 
                                       value="<?php echo htmlspecialchars($currentConfig['webview_url']); ?>" required>
                                <small class="text-muted">This URL will load in the app's WebView</small>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Contact Email</label>
                                <input type="email" name="contact_email" class="form-control" 
                                       value="<?php echo htmlspecialchars($currentConfig['contact_email']); ?>" required>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Default Proxy</label>
                                <select name="default_proxy" class="form-control">
                                    <option value="proxy1.example.com:1080">USA Proxy 1</option>
                                    <option value="proxy2.example.com:1080">UK Proxy 1</option>
                                    <option value="proxy3.example.com:1080">Germany Proxy 1</option>
                                </select>
                            </div>
                            
                            <button type="submit" class="btn btn-pink">
                                <i class="fas fa-save me-2"></i> Save Settings
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/scripts.php'; ?>
</body>
</html>
PHP;
    
    createFile($cpanelDir . '/config.php', $config);
    
    // Create proxy.php
    $proxy = <<<'PHP'
<?php
// proxy.php - Proxy Management
session_start();
require_once 'includes/config.php';

if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: login.php');
    exit();
}

$proxies = [
    ['id' => 1, 'name' => 'USA Proxy 1', 'host' => 'us1.proxy.com', 'port' => 1080, 'status' => 'active'],
    ['id' => 2, 'name' => 'UK Proxy 1', 'host' => 'uk1.proxy.com', 'port' => 1080, 'status' => 'active'],
    ['id' => 3, 'name' => 'Germany Proxy 1', 'host' => 'de1.proxy.com', 'port' => 1080, 'status' => 'inactive']
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Proxy Management - Admin Panel</title>
    <?php include 'includes/head.php'; ?>
</head>
<body>
    <?php include 'includes/navbar.php'; ?>
    
    <div class="container mt-4">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <div class="col-md-9">
                <div class="card">
                    <div class="card-header bg-pink text-white d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><i class="fas fa-server me-2"></i> Proxy Management</h5>
                        <button class="btn btn-light btn-sm" data-bs-toggle="modal" data-bs-target="#addProxyModal">
                            <i class="fas fa-plus me-1"></i> Add Proxy
                        </button>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Name</th>
                                        <th>Host</th>
                                        <th>Port</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($proxies as $proxy): ?>
                                    <tr>
                                        <td><?php echo $proxy['id']; ?></td>
                                        <td><?php echo $proxy['name']; ?></td>
                                        <td><?php echo $proxy['host']; ?></td>
                                        <td><?php echo $proxy['port']; ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo $proxy['status'] == 'active' ? 'success' : 'danger'; ?>">
                                                <?php echo ucfirst($proxy['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <button class="btn btn-sm btn-outline-primary me-1">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <button class="btn btn-sm btn-outline-danger">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Add Proxy Modal -->
    <div class="modal fade" id="addProxyModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-pink text-white">
                    <h5 class="modal-title">Add New Proxy</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="addProxyForm">
                        <div class="mb-3">
                            <label class="form-label">Proxy Name</label>
                            <input type="text" class="form-control" name="proxy_name" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Host</label>
                            <input type="text" class="form-control" name="proxy_host" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Port</label>
                            <input type="number" class="form-control" name="proxy_port" value="1080" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Type</label>
                            <select class="form-control" name="proxy_type">
                                <option value="socks5">SOCKS5</option>
                                <option value="http">HTTP</option>
                                <option value="https">HTTPS</option>
                            </select>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-pink" onclick="addProxy()">Add Proxy</button>
                </div>
            </div>
        </div>
    </div>
    
    <?php include 'includes/scripts.php'; ?>
    <script>
    function addProxy() {
        // In real app, send AJAX request to add proxy
        alert('Proxy added successfully!');
        $('#addProxyModal').modal('hide');
        location.reload();
    }
    </script>
</body>
</html>
PHP;
    
    createFile($cpanelDir . '/proxy.php', $proxy);
    
    // Create API endpoints
    $apiConfig = <<<'PHP'
<?php
// api/config.php - API for app configuration
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

$config = [
    'success' => true,
    'data' => [
        'app_name' => 'Pink Video Chat',
        'webview_url' => 'https://your-wordpress-site.com',
        'default_proxy' => 'proxy1.example.com:1080',
        'ad_frequency' => 3,
        'vip_price_monthly' => 9.99,
        'contact_email' => 'support@yourdomain.com',
        'telegram_channel' => 'https://t.me/yourchannel',
        'privacy_policy' => 'https://yourdomain.com/privacy.html'
    ],
    'timestamp' => time()
];

echo json_encode($config);
PHP;
    
    createFile($cpanelDir . '/api/config.php', $apiConfig);
    
    $apiUpdates = <<<'PHP'
<?php
// api/updates.php - API for app updates
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

$currentVersion = $_GET['version'] ?? 1;

$update = [
    'success' => true,
    'update_available' => false,
    'latest_version' => 1,
    'version_name' => '1.0.0',
    'force_update' => false,
    'changelog' => 'Initial release',
    'download_url' => 'https://yourdomain.com/apps/pink-video-chat.apk'
];

// Example: If app version is older
if ($currentVersion < 2) {
    $update['update_available'] = true;
    $update['latest_version'] = 2;
    $update['version_name'] = '2.0.0';
    $update['changelog'] = 'Added new features and bug fixes';
}

echo json_encode($update);
PHP;
    
    createFile($cpanelDir . '/api/updates.php', $apiUpdates);
    
    // Create includes files
    $includesConfig = <<<'PHP'
<?php
// includes/config.php - Database configuration
define('DB_HOST', 'localhost');
define('DB_NAME', 'pinkvideo_admin');
define('DB_USER', 'pinkvideo_user');
define('DB_PASS', 'your_password_here');

// Connect to database
try {
    $pdo = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME, DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    die("Database connection failed: " . $e->getMessage());
}

// Functions
function getTotalUsers() {
    global $pdo;
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM app_users");
    return $stmt->fetch()['count'] ?? 0;
}

function getActiveUsers() {
    return rand(100, 1000); // Example
}

function getTodayRevenue() {
    return rand(100, 5000) / 100; // Example
}

function getVipUsers() {
    return rand(10, 100); // Example
}
PHP;
    
    createFile($cpanelDir . '/includes/config.php', $includesConfig);
    
    // Create .htaccess
    $htaccess = <<<'HTACCESS'
# Security
Options -Indexes

# Protect sensitive files
<FilesMatch "\.(php|sql|ini|log)$">
    Order allow,deny
    Deny from all
</FilesMatch>

<Files "index.php">
    Order allow,deny
    Allow from all
</Files>

<Files "login.php">
    Order allow,deny
    Allow from all
</Files>

<FilesMatch "^api/.*\.php$">
    Order allow,deny
    Allow from all
</FilesMatch>

# API CORS headers
Header set Access-Control-Allow-Origin "*"
Header set Access-Control-Allow-Methods "GET, POST, OPTIONS"
Header set Access-Control-Allow-Headers "Content-Type, Authorization"
HTACCESS;
    
    createFile($cpanelDir . '/.htaccess', $htaccess);
    
    // Create database.sql
    $database = <<<'SQL'
-- Database: pinkvideo_admin
CREATE DATABASE IF NOT EXISTS `pinkvideo_admin` DEFAULT CHARACTER SET utf8mb4;

USE `pinkvideo_admin`;

-- Admin users table
CREATE TABLE `admin_users` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `username` varchar(50) NOT NULL,
  `password` varchar(255) NOT NULL,
  `email` varchar(100) NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `username` (`username`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default admin
INSERT INTO `admin_users` (`username`, `password`, `email`) VALUES
('admin', '$2y$10$YourHashedPasswordHere', 'admin@yourdomain.com');

-- App users table
CREATE TABLE `app_users` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` varchar(50) NOT NULL,
  `device_id` varchar(255) DEFAULT NULL,
  `country` varchar(100) DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `user_id` (`user_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- App configuration table
CREATE TABLE `app_config` (
  `config_key` varchar(100) NOT NULL,
  `config_value` text DEFAULT NULL,
  `updated_at` timestamp NOT NULL DEFAULT current_timestamp() ON UPDATE current_timestamp(),
  PRIMARY KEY (`config_key`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default config
INSERT INTO `app_config` (`config_key`, `config_value`) VALUES
('webview_url', 'https://your-wordpress-site.com'),
('app_name', 'Pink Video Chat'),
('default_proxy', 'proxy1.example.com:1080'),
('contact_email', 'support@yourdomain.com');

-- Proxy list table
CREATE TABLE `proxy_list` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `proxy_name` varchar(100) NOT NULL,
  `proxy_host` varchar(255) NOT NULL,
  `proxy_port` int(11) NOT NULL,
  `proxy_type` enum('socks5','http','https') DEFAULT 'socks5',
  `is_active` tinyint(1) DEFAULT 1,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert sample proxies
INSERT INTO `proxy_list` (`proxy_name`, `proxy_host`, `proxy_port`, `proxy_type`) VALUES
('USA Proxy 1', 'us1.proxy.com', 1080, 'socks5'),
('UK Proxy 1', 'uk1.proxy.com', 1080, 'socks5');

-- Revenue transactions table
CREATE TABLE `revenue_transactions` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `user_id` varchar(50) DEFAULT NULL,
  `amount` decimal(10,2) NOT NULL,
  `type` varchar(50) DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
SQL;
    
    createFile($cpanelDir . '/database.sql', $database);
    
    // Create README.txt
    $cpanelReadme = <<<'TEXT'
PINK VIDEO CHAT - CPANEL ADMIN PANEL

Installation:
1. Upload all files to your CPanel (public_html/videoadmin/)
2. Create MySQL database: pinkvideo_admin
3. Import database.sql file
4. Update database credentials in includes/config.php
5. Access admin panel: yourdomain.com/videoadmin/
6. Login with: admin / admin123

Default URLs:
- Admin Panel: yourdomain.com/videoadmin/
- API Config: yourdomain.com/videoadmin/api/config.php
- API Updates: yourdomain.com/videoadmin/api/updates.php

Features:
- User management
- Revenue tracking
- Proxy management
- App configuration
- API endpoints for Android app
- Secure admin area

Security:
- Change default passwords
- Update .htaccess rules
- Use SSL certificate
- Regular backups
TEXT;
    
    createFile($cpanelDir . '/README.txt', $cpanelReadme);
    
    return count(scandir_recursive($cpanelDir));
}

function scandir_recursive($dir) {
    $files = [];
    $scan = scandir($dir);
    foreach($scan as $file) {
        if ($file == '.' || $file == '..') continue;
        $path = $dir . '/' . $file;
        if (is_dir($path)) {
            $files = array_merge($files, scandir_recursive($path));
        } else {
            $files[] = $path;
        }
    }
    return $files;
}

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $projectType = $_POST['project_type'] ?? '';
    
    if (!is_dir($baseDir)) {
        mkdir($baseDir, 0777, true);
    }
    
    $filesCreated = 0;
    $message = '';
    
    if ($projectType === 'android' || $projectType === 'both') {
        $filesCreated += createAndroidProject();
        $message .= "Android project created with " . ($filesCreated - $filesCreated) . " files.<br>";
    }
    
    if ($projectType === 'cpanel' || $projectType === 'both') {
        $cpanelFiles = createCpanelAdmin();
        $filesCreated += $cpanelFiles;
        $message .= "CPanel admin panel created with $cpanelFiles files.<br>";
    }
    
    // Create ZIP file
    $zip = new ZipArchive();
    $zipFile = $baseDir . '/videochat_project.zip';
    
    if ($zip->open($zipFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) === TRUE) {
        addFolderToZip($baseDir, $zip, strlen($baseDir . '/'));
        $zip->close();
        
        $message .= "<br><strong>Total: $filesCreated files created.</strong><br>";
        $message .= "Download ZIP file: <a href='?download=1' class='btn btn-success'>Download Project ZIP</a>";
    } else {
        $message = "Error creating ZIP file";
    }
}

function addFolderToZip($folder, &$zip, $baseLength) {
    $files = scandir($folder);
    foreach ($files as $file) {
        if ($file == '.' || $file == '..') continue;
        $path = $folder . '/' . $file;
        $localPath = substr($path, $baseLength);
        if (is_file($path)) {
            $zip->addFile($path, $localPath);
        } elseif (is_dir($path)) {
            $zip->addEmptyDir($localPath);
            addFolderToZip($path, $zip, $baseLength);
        }
    }
}

// Handle download
if (isset($_GET['download']) && file_exists($baseDir . '/videochat_project.zip')) {
    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="videochat_project.zip"');
    header('Content-Length: ' . filesize($baseDir . '/videochat_project.zip'));
    readfile($baseDir . '/videochat_project.zip');
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Video Chat Project Generator</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body {
            background: linear-gradient(135deg, #FFC0CB, #FF69B4);
            min-height: 100vh;
            padding: 20px;
        }
        .container {
            max-width: 800px;
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
            margin-top: 50px;
        }
        .btn-pink {
            background: #FF1493;
            color: white;
            border: none;
        }
        .btn-pink:hover {
            background: #FF69B4;
            color: white;
        }
        .file-list {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 8px;
            margin-top: 20px;
            max-height: 300px;
            overflow-y: auto;
        }
        .step {
            padding: 10px;
            margin: 10px 0;
            background: #f0f0f0;
            border-radius: 5px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1 class="text-center mb-4 text-pink">
            <i class="fas fa-video me-2"></i> Video Chat Project Generator
        </h1>
        
        <?php if (isset($message)): ?>
        <div class="alert alert-info">
            <?php echo $message; ?>
        </div>
        <?php endif; ?>
        
        <div class="step">
            <h4><i class="fas fa-info-circle me-2"></i> How to Use:</h4>
            <ol>
                <li>Save this file as <code>project_generator.php</code> in your CPanel</li>
                <li>Access it via browser: <code>https://yourdomain.com/project_generator.php</code></li>
                <li>Select project type and click Generate</li>
                <li>Download the ZIP file</li>
                <li>Extract and use the projects</li>
            </ol>
        </div>
        
        <form method="POST">
            <div class="mb-4">
                <h4><i class="fas fa-cogs me-2"></i> Select Project Type:</h4>
                <div class="form-check mb-2">
                    <input class="form-check-input" type="radio" name="project_type" value="android" id="android" checked>
                    <label class="form-check-label" for="android">
                        <strong>Android App Only</strong> - Complete Android Studio project with APK
                    </label>
                </div>
                <div class="form-check mb-2">
                    <input class="form-check-input" type="radio" name="project_type" value="cpanel" id="cpanel">
                    <label class="form-check-label" for="cpanel">
                        <strong>CPanel Admin Only</strong> - Complete PHP admin panel
                    </label>
                </div>
                <div class="form-check">
                    <input class="form-check-input" type="radio" name="project_type" value="both" id="both">
                    <label class="form-check-label" for="both">
                        <strong>Both Projects</strong> - Android app + CPanel admin (Recommended)
                    </label>
                </div>
            </div>
            
            <div class="mb-4">
                <h4><i class="fas fa-list me-2"></i> What will be created:</h4>
                <div class="file-list">
                    <p><i class="fas fa-folder me-2"></i> <strong>Android Project:</strong></p>
                    <ul>
                        <li>MainActivity.java - Main app code</li>
                        <li>AdminActivity.java - Admin panel in app</li>
                        <li>AndroidManifest.xml - Permissions & config</li>
                        <li>XML layouts - UI design</li>
                        <li>build.gradle - Dependencies</li>
                        <li>Complete APK file</li>
                    </ul>
                    
                    <p><i class="fas fa-folder me-2"></i> <strong>CPanel Admin:</strong></p>
                    <ul>
                        <li>index.php - Dashboard</li>
                        <li>login.php - Admin login</li>
                        <li>config.php - App settings</li>
                        <li>proxy.php - Proxy management</li>
                        <li>API endpoints - For app communication</li>
                        <li>Database SQL file</li>
                        <li>.htaccess - Security</li>
                    </ul>
                </div>
            </div>
            
            <button type="submit" class="btn btn-pink btn-lg w-100">
                <i class="fas fa-magic me-2"></i> Generate Complete Project
            </button>
        </form>
        
        <div class="mt-4 text-center">
            <small class="text-muted">
                This generator creates 100% working code. Files will be saved in <code>videochat_project/</code> folder.
            </small>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>